/*!
 *
 * Jquery Mapael - Dynamic maps jQuery plugin (based on raphael.js)
 * Requires Mapael
 *
 * Map of World by country
 * Miller projection
 *
 * @source http://code.highcharts.com/mapdata/
 *
 * @deprecated : this map will be definitely moved to 'mapael-maps' repository starting from the next major release (3.0.0).
 * You can use instead https://github.com/neveldo/mapael-maps/blob/master/world/world_countries_miller.js
 */
(function (factory) {
    if (typeof exports === 'object') {
        // CommonJS
        module.exports = factory(require('jquery'), require('jquery-mapael'));
    } else if (typeof define === 'function' && define.amd) {
        // AMD. Register as an anonymous module.
        define(['jquery', 'mapael'], factory);
    } else {
        // Browser globals
        factory(jQuery, jQuery.mapael);
    }
}(function ($, Mapael) {

    "use strict";

    $.extend(true, Mapael, {
        maps: {
            world_countries_miller: {
                /*
                 * map-specific constants
                 * (set by user)
                 */
                // Width of the map, in pixel
                width : 700.9375,
                // Height of the map, in pixel
                height : 337.375,
                // Longitude of the left side of the map, in degree
                leftLongitude: -175.0,
                // Longitude of the right side of the map, in degree
                rightLongitude: 190.0,
                // Latitude of the top of the map, in degree
                topLatitude: 84.0,
                // Latitude of the bottom of the map, in degree
                bottomLatitude: -56.0,
                /*
                 * Transform a longitude coordinate into projection-specific x' coordinate
                 * Note: this is not in pixel
                 *
                 * @param lon longitude value in degree
                 * @return x' projection-specific value
                 */
                _projectLongitude: function (lon) {
                    // Compute longitude in radian
                    return lon * Math.PI / 180;
                },
                /*
                 * Transform a latitude coordinate into projection-specific y' coordinate
                 * Note: this is not in pixel
                 *
                 * @param lat latitude value in degree
                 * @return y' projection-specific value
                 */
                _projectLatitude: function (lat) {
                    // Compute latitude in radian
                    var latRad = lat * Math.PI / 180;
                    return 1.25 * Math.log( Math.tan( Math.PI / 4 + 0.4 * latRad ) );
                },
                /*
                 * Get x,y point from lat,lon coordinate
                 *
                 * Principle:
                 *      (lat, lon) are inputs
                 *      Projection(lat, lon) = (x', y')
                 *      Transformation(x', y') = (x, y)
                 *
                 * Source: http://jkwiens.com/2009/01/23/miller-projection/
                 *
                 * @param lat latitude value in degree
                 * @param lon longitude value in degree
                 * @return {x, y} coordinate in pixel
                 */
                getCoords: function (lat, lon) {
                    var self = this;

                    // Project map boundaries with projection (only once for performance)
                    if (self._xLeftPrime === undefined) self._xLeftPrime = self._projectLongitude(self.leftLongitude);
                    if (self._xRightPrime === undefined) self._xRightPrime = self._projectLongitude(self.rightLongitude);
                    if (self._yTopPrime === undefined) self._yTopPrime = self._projectLatitude(self.topLatitude);
                    if (self._yBottomPrime === undefined) self._yBottomPrime = self._projectLatitude(self.bottomLatitude);

                    // Compute x' and y' (projection-specific value)
                    var xPrime = self._projectLongitude(lon);
                    var yPrime = self._projectLatitude(lat);

                    // Compute x and y
                    var x = (xPrime - self._xLeftPrime) * (self.width / (self._xRightPrime - self._xLeftPrime));
                    var y = (self._yTopPrime - yPrime) * (self.height / (self._yTopPrime - self._yBottomPrime));

                    return {x: x, y: y};
                },
                elems: {
                    "FO" : "m 322.53,80.19 0.9,0.8 0,0.3 -1,-0.7 z",
                    "UM" : "m 26.03,206.59 0,0 -0.1,0 0.1,0 z",
                    "US" : "m 100.93,117.39 0.2,0.1 -0.6,-0.2 0,-0.3 z m 0.2,-0.6 0.2,0.1 -0.3,0.2 -0.3,-0.1 z m -19,-15.8 0.2,0 0,0.4 -0.3,-0.4 z m -0.1,-0.7 0.3,0.3 -0.2,0.2 -0.2,-0.1 z m -0.5,0 0.8,1.3 -0.9,-0.7 -0.1,-0.7 z m 3.3,0.1 -0.1,0.3 -0.4,0 0.1,-0.7 z m -3.7,-0.5 0.1,0.3 -0.4,0 0,-0.3 z m 2.8,-0.3 0.3,0.4 -0.2,0.4 -0.2,-0.6 z m -3.5,-0.1 0.4,-0.1 0,0.3 -0.4,0.3 z m 0.7,0.1 -0.1,0 -0.2,-0.3 0.3,0 z m -1.1,-1.3 0.3,-0.4 0.8,-0.2 -0.2,0.3 z m 2.6,-0.8 0.8,0.3 -0.4,0.5 -0.7,-0.4 z m -0.3,-0.6 0,0.4 -0.8,-0.2 0.3,-0.3 z m 1,0.2 0.1,0.6 -0.5,-0.4 0,-0.6 z m -1.4,-1.2 0.6,0.6 -0.4,0.2 -0.3,0 z m -1.7,-0.8 1.5,0.5 -0.3,1.2 -1,0 z m -3.5,-0.4 -0.1,0.6 -0.4,0.1 0.2,-0.5 z m -1.5,-2.1 -0.2,0.4 -0.2,-0.2 0.2,-0.5 z m 91.6,25.5 -0.2,0.4 -1,0.5 -0.1,-0.2 z m 3,6.8 -0.2,0.6 -0.4,0.7 -0.2,-0.4 z m 2.8,-1 0,0.4 -0.3,0 0.2,-0.4 z m 3.7,-0.7 0.1,0.3 -0.6,0 0.3,-0.2 z m -96.8,-24.9 -0.6,-2.5 0.8,-0.1 0.6,0.7 z m -61.3,4.8 -0.9,1.5 -2.4,0.4 -2.2,1.3 1.2,-1.5 2.3,0 0,-1 z m 59.3,-7.9 -0.4,-1.4 1,0.8 0.4,3 z m -0.7,-2.2 -0.3,1.3 -1.4,-1.2 1.1,-1.3 z m 1.1,-1.2 1.4,0.4 0.5,2.3 -1,-0.2 z m -66.2,-14.8 1.6,0.2 -2,1.1 -1.9,-1.3 -2.2,-0.4 3,-0.6 z m 89,39.1 -0.1,0.1 0,-0.1 z m 45.7,49.9 -0.1,-0.8 -0.5,-0.2 -1.5,-2.5 -1.5,-2.5 -0.2,-0.1 -0.1,-0.1 -2.1,-0.7 -1.6,1.9 -2.6,-1.4 -0.7,-1.8 -3.2,-2.6 -3.2,0 0,0.9 -5.5,0 -7.2,-2.4 0.2,-0.5 -4.6,0.4 -0.3,-1.2 -2.4,-2 -2.5,-0.1 -1.5,-1.1 -3.4,-5.7 -0.2,-1.3 -2.3,-2.5 -0.3,-2 -1,-1.4 0.6,-2.3 -0.9,-3.2 0.8,-2.4 0.5,-4 -1.3,-6.7 3.6,0.6 -0.1,-0.2 0.7,0.6 -0.7,-0.7 0,0 0.3,-0.4 -0.3,0.3 0,-1.8 52.8,0 4.8,1.2 2.3,1.2 3.6,0.1 -2.8,1.2 -2,1.7 2.5,-0.2 0.6,0.7 3.5,-1.6 0.1,0.1 -0.1,-0.1 1.1,-0.6 0.6,0.1 -1.1,0.9 1.8,1.4 4.6,-0.7 0.9,0.8 0.7,-0.2 0,0.5 -1.2,1.1 -4.3,-0.1 -2,3.2 1.4,-0.6 -1.1,3.6 0.1,2.3 1,1.5 1.3,-0.6 0.8,-2.5 -0.6,-1.5 0.5,-2.5 3.2,-2.7 2.1,1.1 0.3,2.4 -1,0.8 2.3,0.2 0.5,2.1 -0.2,0.9 -0.3,0.2 -0.3,-0.3 -0.5,0.8 -0.7,1.4 1.8,0.7 1.6,-0.3 4.8,-2.4 0.5,-0.8 -0.3,-0.4 0,-0.4 4.5,-0.1 1.8,-2.7 1,-0.9 1,-0.3 0,0 6.2,-0.1 2.3,-2 0.5,-2 1.5,-1.8 2.7,0.9 0,3.2 0.4,0.3 0.4,0 -0.1,0.8 0.6,0.2 0.3,0.9 -2.4,1 0.1,0.1 -0.4,0.4 0,-0.3 0,0 -0.7,0.3 0.3,-0.1 0,0.4 -0.3,0 -0.1,-0.3 -0.4,0.2 -2.9,1.8 -0.1,3.7 -4.4,1.1 -2.2,1.4 -0.1,1.9 -2.9,4 -1.5,-2.8 0.3,3 -0.4,2.2 0.9,-0.1 0.5,2.8 -1.5,2.2 -1.2,-0.1 -1.6,1.8 -2.3,1.4 -3.8,3.1 -0.6,2.1 1.8,4.7 0.9,3.4 -0.6,3.4 -1.4,0 -2,-2.7 -1.3,-2.9 0.4,-2.1 -2.7,-2.5 -1.8,1 -2.1,-1.4 -5.3,-0.3 -1.5,0.6 0.9,2.4 -1.6,-0.9 -1.3,0.8 -1.9,-1.4 -1.3,0.4 -3.4,-0.3 -4.6,2.8 -2,3.1 0.8,1.7 -2.1,-0.2 z m -65.4,-67.3 -0.3,0.2 -0.2,-0.4 0.5,0 0,0 0.3,-1.3 2.3,3 -0.1,1.4 -1.6,-1.8 -1,-0.7 z m -43.4,79.5 -0.5,-0.5 0,0 0.5,0.2 0,0.3 0,0 z m 0,0 2.3,2 -2.1,0.9 -0.2,-2.9 0,0 z m 47.3,-77.4 -0.1,0.3 -0.3,-0.2 0.4,-0.1 0,0 z m -5.8,-8.9 0.1,0.1 -0.6,-0.1 0,-0.2 -0.3,-0.2 -3.7,0.5 -3.1,-2.4 -3,-1.3 -5.6,-1.7 -2.8,-0.1 -3.5,-0.9 0,-1.1 -3.5,0 0.4,2.5 -3.2,0 -0.7,0.9 -4.1,1.5 1.3,-1.3 -1.3,-0.4 0.9,-2.8 1.8,-0.9 -0.2,-0.9 -3.4,2.4 -0.7,1.7 -2.7,1.5 1.5,1.3 -1.9,2.4 -3.6,1.6 -2.1,2.7 -0.9,-0.5 -2.1,2.2 -1.9,0.1 -3.2,1.2 -1.6,1.4 -4.7,1.2 0.5,-1.3 2.4,-0.6 3.7,-2.2 2.2,0.3 -0.3,-1.2 2.8,-1.5 2.2,-2 1.3,-3.7 -3.7,1.3 -2.5,-1.8 -2.7,1.4 0.1,-2.6 -1.2,-1.8 -2.5,0.9 -2.6,-1.5 -1.1,1 -2.1,0.4 -1.5,-1 3.4,-0.3 2.1,-1.6 -3,-1.9 0.7,-1.5 3.1,-3.7 1.5,0.7 1.8,-1.2 3.1,-1 -1.2,-2.1 0.5,-1.6 -3.8,1.7 -5.8,-0.6 -3.6,-3.5 5.3,-2.3 3,-0.5 0.1,1.6 3.5,0.3 0.3,-3.3 -3.9,-0.3 -0.8,-1.6 -4.8,-2.4 0.8,-1.8 3.6,-0.2 2.4,-1.5 -0.1,-0.9 2.4,-2.4 2,0 2.3,-1.6 3.1,-0.1 2.3,-1.7 3.3,0.7 1.9,1.2 3.4,-0.3 0.7,1.4 5.3,-0.3 4,1.3 4.8,0.7 2.5,-0.5 4.4,1.6 0,29.2 3.5,0.6 3,2.4 0.4,1 2,-2 1.7,-0.6 1,1.5 3,2.5 3,5 3.4,1.8 0.1,1.7 0,0 -1.8,1.3 0.1,-2.4 -0.3,-0.1 -0.3,-0.6 -0.4,0.4 -0.6,-0.2 -0.5,-1.3 -2.1,-2 -2.2,-3.1 z m -38.1,3.7 -1.1,-1.2 3,-2 1.7,-1.6 0.4,2.8 -2.9,1.7 -3.7,3.4 z m -3.9,82.1 -3,-1 2.1,0.2 0.9,0.8 z m 47.8,-76.9 0.1,-0.2 0,0.2 -0.1,0 0,0 z",
                    "JP" : "m 606.33,125.09 0,-0.1 0.6,0.4 -0.3,0.1 z m -15.5,25.1 -1.3,0.7 -2.1,-0.3 2.8,-3.2 3.7,-0.5 2.2,0.3 3,-4.4 -0.5,1.4 2.4,-0.6 0.4,-1.6 1.8,-0.6 1.1,-2.6 -0.4,-1.2 1.1,-3.2 0.7,1 1.3,-0.7 1.2,3.9 -1.8,2.3 -0.1,2.2 -0.9,2.4 0.5,1.2 -1.7,1.8 -0.3,-1.4 -3,1.8 -3.2,0.3 0.7,0.7 -1.8,1.4 -1.7,-0.6 0.6,-1.7 -3.2,0.3 -1.5,0.9 1.1,0.3 1.3,-1.2 0.9,0.5 -0.9,2.3 -0.9,-0.6 -2.2,1.7 -0.5,-1.6 z m -6.8,2 -0.4,-2 1.3,1.7 1.2,-1.1 2.4,0.6 0.6,1.3 -1.4,3.4 -2.1,0.2 -0.1,-3.3 z m 21.7,-19.1 -1,1 -0.6,-2.9 1.1,-1.6 1.9,0 0.9,-5.1 3.6,3.2 2.4,0.9 0.9,1.4 -2.6,0.5 -1.8,2.4 -2.7,-1.5 -2.9,0.7 z",
                    "SC" : "m 442.43,226.39 0.2,0.3 -0.1,0.2 -0.5,-0.7 z",
                    "IN" : "m 504.93,175.49 0.1,0 0,0.5 -0.2,-0.1 z m 0.1,-12.1 1.3,-0.3 0.2,1.4 1.7,1.2 4.4,-0.4 -0.9,-1.7 1.9,-0.4 3.9,-2.9 1.2,0.6 1.6,-0.7 2.3,2.4 -1.2,1.8 -3.2,1.8 0,1.5 -1.8,3.8 -1.5,-0.3 -0.2,3.3 -1.2,0.8 -0.8,-3.5 -1.1,1.5 -0.8,-1.3 2.5,-2.7 -0.8,-0.5 -3.2,0.1 -1,-1.7 -2.7,-1 -0.7,1.2 1.6,1.5 -1.7,1 1.4,0.7 -0.4,1.4 1,3 -1.6,1.1 -0.1,-1 -1.9,1 -1.6,3.2 -1.7,0.1 -2.6,3.2 -3.5,2.5 0.1,0.8 -3.9,1.8 -0.5,1.3 0.6,3.2 -0.9,2.6 0,3.3 -4.5,4.3 -1.9,-1.6 -1.3,-4.2 -2,-3.4 -0.9,-3.3 -1.7,-3 -1.5,-7.4 0.4,-1.3 -1,-3.6 -0.5,2.2 -2.1,1 -1.3,-0.4 -2.1,-2.1 1.8,-1.2 -2.9,-1.2 -0.9,-1.4 1.3,-0.8 2.1,0.2 2.1,-1.1 -3,-4.3 1.9,-2.5 2.7,0.2 2,-2.3 0.9,-1.8 2.2,-2.5 -0.1,-1.3 1.5,-0.9 -2.4,-2.2 -0.5,-2.5 1.1,-0.9 2.8,0.6 2.4,-1.3 1.4,-0.9 0.9,1.8 1.3,0.7 -0.3,1.7 1.4,1.7 -2.2,0.5 0.7,2.5 4.4,2.4 -1.3,0.9 -0.6,1.9 5.2,2.8 3.5,0.3 2.4,1.6 2.9,0.4 1.6,-0.7 z",
                    "FR" : "m 228.83,121.39 -0.4,0.1 -0.1,-0.8 0.2,0.5 z m 125.6,9.9 -0.5,2.9 -0.8,-0.5 -0.1,-2.3 z m 301.9,127.2 -0.6,2.6 -3,-1.6 -1.7,-1.7 -0.4,-1.4 4.2,3.2 z m -187.7,60.9 2.7,0.3 -0.8,1.5 -2.5,0 z m -253.1,-136.4 0.1,-0.1 0,0.2 z m 135.1,-54.4 0,0.1 -0.1,0 0,0 -0.1,0.1 -2.4,1.4 -4.2,-1 -1.7,1.4 0.3,1.2 -1.3,0.2 -1,-0.3 -0.1,0 0,0 -0.5,-0.1 0.1,-0.3 -0.6,0.1 -2.8,-0.2 -3.4,-1.7 0.6,-0.4 0.9,-5.8 -1.8,-2.4 -1.6,-1.1 -3.6,-1.4 -0.1,-1.2 3.2,-0.8 0.7,0.9 2.5,-0.3 -1,-2.7 3.3,1.1 0.7,-1 2.7,-1.3 0,-1.7 1.8,-0.5 3.1,2.8 3.2,1.1 0.3,0.2 0.7,0 3.6,1.2 -1.2,3.4 -2.8,2.4 0.2,0.7 0.1,-0.2 0.2,0 0.8,-0.1 -0.1,0.7 0.5,0.5 -0.8,1.9 z m -118,79 0.4,-0.8 1.9,0.6 2.4,2.7 -2.5,3.5 -3.1,-0.3 1.2,-2.1 -0.9,-2.8 z",
                    "FM" : "m 639.23,204.49 0.2,0.1 0,0.2 -0.2,0 z",
                    "CN" : "m 544.43,180.89 1.7,-1.7 2.1,-0.3 0.6,0.9 -1.1,2 -1.4,1 -1.9,-0.7 z m -37.9,-16.4 -0.2,-1.4 -1.3,0.3 -3,-0.5 -1.1,0.4 -1.7,-0.9 -1.9,-1.9 -1.1,0.1 -2.8,-2.4 -2,0.3 -4.4,-2.4 -0.7,-2.5 2.2,-0.5 -1.4,-1.7 0.3,-1.7 -1.3,-0.7 -0.9,-1.8 -0.8,0.1 -1.1,-0.4 -1.6,-1.8 -2.7,-1.2 -0.3,-0.2 1,-0.3 -0.3,-2.8 -1.8,-0.1 -0.3,-2 0.6,-1.3 3,-1.3 0.3,0.7 2.1,-1.4 2.5,-0.3 0.5,-0.7 3.5,-1.9 1.2,-2.2 -1.4,-4.4 4.2,-1.2 1.4,-4 3.3,0.9 1.5,-0.5 0.5,-3.3 3,-1.7 0.3,-0.1 0.6,-0.1 0.3,1.4 2.1,1.5 2.4,0.8 1.4,2.6 -0.3,3.3 5,0.6 3.6,1.5 1.9,3.7 10.1,0.5 0.7,0.7 3.3,1 2.8,0 3,-1.2 5.2,-0.4 4.7,-2.9 -1,-1.7 1.1,-1.5 3.1,0.8 1.8,-1.5 2.1,-0.2 1.8,-2 3.2,-1 2.7,0.3 0.4,-0.8 -2.6,-2.6 -1.5,0 -0.1,0.1 -0.3,0.4 -3.2,0.2 -0.7,-1.1 2.3,-4.1 2,0.7 2.8,-1.2 -0.1,-0.9 1.8,-3.3 1.3,-1.1 -1.4,-1.7 1.5,-1.3 4.7,-0.8 4.7,1.5 4.5,8.8 2.5,0.3 2.8,2.7 0.4,1.6 2.7,-0.1 3.7,-1.7 0.7,1.7 -1.2,1.1 -0.5,2.4 -2,2.8 -0.5,-0.6 -1.1,0.2 -2,0.9 0.6,1.9 -0.5,2.8 -0.9,0.7 -1.3,-1 -0.3,1.2 -3.2,1.1 0.2,1.4 -2.9,-0.7 -1.2,1.8 -3.1,1.8 -3.6,1.5 -2.6,1.5 0.3,-1.4 1.9,-2.5 -2.1,-1 -1.4,1.7 -1.8,0.6 -1,1.6 -2.4,0.2 0.7,2.1 1.4,0 1.2,2.3 2.4,-1.6 1.7,0.9 2.1,0.1 -3.7,2 -3,3.1 2,1.6 1.2,3.5 1.8,1.5 0.3,2.2 -1.9,0.8 2.7,1.5 -1.5,1.1 0.1,2 -1.3,0.7 -2.9,4.1 0.6,1.2 -1.8,0.6 -4.5,4.4 -3.3,0.2 -1.1,1.2 -1.1,-1.6 -0.1,1.8 -5.5,1.5 -1.5,2.2 -0.3,-2.4 -2.5,-0.8 -0.9,0.8 -2.7,-1.8 0.4,-0.8 -2.8,-1.2 -2.6,1.7 -2.8,-0.5 -0.7,0.7 -1.2,0.3 0.4,2.3 -1,-0.9 -1.9,0.3 -0.5,-1.3 -1.5,-0.2 0.7,-1.8 -1.2,-0.2 -0.6,-1.8 -1.8,0.4 0.1,-2.4 1.9,-1.7 0,-3.5 -2.2,-1.9 -0.4,0.6 -2.3,-2.4 -1.6,0.7 -1.2,-0.6 -3.9,2.9 -1.9,0.4 -2.7,-1.3 z",
                    "PT" : "m 322.13,143.49 -3.1,0.4 0.5,-2.6 -1.4,-1.1 1.5,-4.7 -0.1,-2.7 4.2,0.1 0.7,0.8 -1.4,1.3 -0.1,4.5 z",
                    "SW" : "m 185.73,187.29 0,0 0,0 z",
                    "SH" : "m 561.73,188.69 0,0 0,0 0,0 z",
                    "BR" : "m 241.13,217.99 2.5,0.3 -0.8,2.2 -3.4,0.6 -0.4,-2.2 z m -7.3,66.1 0.8,-1.6 -0.1,-0.3 -4.7,-3.8 -0.8,0.5 -1.5,-2.1 -1.5,0.2 3.5,-4.1 3.7,-2.2 -0.1,-3.1 -1.4,-0.1 0.1,-0.1 -0.1,-0.2 0.7,-1.5 0,-1.3 -2.2,-0.2 -0.4,-2.6 -1.5,-1.1 -3,-0.1 0.3,-2.2 -0.7,-1.6 1.2,-3.9 -1.6,-1.8 -0.2,-1.8 -3.2,-0.1 -0.6,-4.8 -2.6,-0.5 -2.4,-1.7 -2.5,-0.4 -1.9,-2.5 -0.1,-2.9 -2.3,0.5 -3.2,2 -2.4,0 -2.1,0.1 0.3,-3 -1.3,0.9 -1.9,0.2 -3.5,-4.8 1.7,-2.2 0.4,-2.4 4.1,-1.8 1.5,0.1 1,-5.3 -1.2,-2.4 0.4,-3.6 3.3,0 1.5,-0.7 0.9,1.6 2.6,1.1 2.7,-1.7 1.5,-1.6 -1.3,-0.1 -1.2,-2.9 5.9,-0.5 1.6,-1.8 1.4,0.2 0.9,2.2 -0.9,2.3 0.5,1.6 1.8,1.3 3.2,-1.6 1.2,0.2 1,-1.1 2.6,0.3 3.1,0.3 2.5,-3.5 1.2,0.3 0.7,3.2 1.5,1 0.2,3.1 -3,0.9 1.4,2.6 2.7,0.3 2.9,-2.7 5.2,1.5 0.7,2.6 1.8,-0.8 4,1.2 3.1,-0.1 2.9,1.6 2.5,2.3 3.1,0.4 1.5,3.9 -1,3.9 -3.1,3 -2.2,3.6 -1.7,1.2 0.2,5.1 -0.6,3.5 -0.9,1.1 -0.1,2.3 -2.7,4.2 0.2,0.9 -2,2 -4.4,0 -5,2.7 -1.7,1.3 -1.5,2.3 0.4,2.7 -0.7,2.3 -2,2 -1,2.1 -1.7,-0.6 -0.4,1.9 -1.5,1.4 -0.8,2.9 -1.4,1.3 -0.3,-0.2 z",
                    "KI" : "m 34.93,213.89 0.2,0.4 0.2,0.2 -0.4,-0.1 z",
                    "PH" : "m 574.43,195.49 1.3,-0.2 -1.5,-1.5 1.7,-0.1 1.1,2.8 -1.3,-0.2 -0.5,1.8 z m -8.8,0.5 0.2,-1.8 0.9,0.5 -1.1,1.3 z m -0.4,0.4 0,0 0.4,-0.4 0,0 0,0 -0.4,0.4 z m 4.6,1.5 0.1,-3.5 2.2,1.5 -0.4,1.1 2.1,-0.8 -0.1,1.8 -1.9,2.5 -1.2,-1.5 0.4,-1.7 z m 4.4,5.8 -1.6,-0.8 -1.8,0.2 -1,2.5 0.2,-2.6 2.5,-1.9 1,1.1 1.7,-1.7 1.4,0.1 0,-2 1.6,2.3 0.5,3 -0.7,1 -0.9,-1.1 -0.8,1.3 0.1,2 -2.2,-1.1 z m 0.3,-12.1 -0.7,3.8 -1.2,-1 -0.5,-2.4 -1.2,-1 0.1,1.6 -1.7,-1.6 -1,1.1 1,0.9 -0.9,1.4 -1.6,-2.5 1.2,-0.1 -1.8,-2.4 -0.5,-3 1.1,0.4 -0.1,-2.8 1.6,-2.9 -0.2,0.8 2.3,0.8 0.5,2.2 -0.7,1.7 -1.2,0.6 0.3,3.4 1.5,-0.3 z m -9.3,4.8 0.3,1.3 -4.6,3.9 4.3,-5.2 z",
                    "MX" : "m 170.23,178.09 -0.5,0.6 0,-0.3 0.1,-0.3 z m -38.2,-2.3 0.3,0.1 0.6,0.8 -0.5,-0.4 z m -10.1,-5.6 0.3,0 0.2,0.3 0,0.1 z m -0.4,-1 -0.1,0.4 -0.1,0 0.1,-0.4 z m -5.8,-6.3 -0.1,0.1 -0.2,-0.2 0.3,-0.5 z m -5.8,-1.7 -0.1,0 -0.1,-0.5 0.2,0.2 z m 10.9,0.4 0.2,-1 0.5,0.4 -0.3,0.5 z m -1.8,-1.7 0.6,0.5 0.1,0.6 -0.9,-0.8 z m 23.5,-0.1 -0.1,0.2 0.4,0 1.5,2.5 1.5,2.5 0.4,0.5 0.2,0.5 1.8,1 2.1,0.2 -1,3 -0.4,4 1.4,3.7 2.5,3.8 2.6,1.2 5,-1 1.8,-1 1.1,-3.7 4.2,-1.2 2.7,0.7 -1.3,2.1 -0.8,3.9 -0.9,-0.5 -1.1,1.2 -0.5,0.1 -3.5,0 -0.9,1.1 2,1.8 -2.6,0.6 -0.9,2.9 -3.2,-3 -1.8,-0.7 -3.3,1.5 -4.2,-1.7 -4.4,-1.4 -1.4,-1.2 -3.3,-0.9 -2.8,-2 -1.4,-2.1 1,-2.1 -1.2,-2.3 -4.8,-5.3 -2.2,-1.1 0,-1.9 -2.1,-1.3 -3,-3.4 -1.9,-3.8 0.1,-0.7 -2.9,-1.3 -0.2,3.3 2,1.9 6,9 -0.2,1.3 1.6,0.2 0.6,1.9 -0.8,0.7 -0.8,-1.4 -3.5,-2.6 -0.2,-2.4 -2.6,-1.4 -2.8,-2.3 1.7,-0.2 0.2,-1.1 -3.1,-2.7 -0.7,-2.2 -2.1,-3.6 4.6,-0.4 -0.2,0.5 7.2,2.4 5.5,0 0,-0.9 3.2,0 3.2,2.6 0.7,1.8 2.6,1.4 1.6,-1.9 z m 7,8.5 -0.4,1 0.2,0.8 0.5,-1.8 z",
                    "ES" : "m 306.83,163.39 -2,-0.6 -1.3,0.1 1.9,-1 z m 23.8,-15.8 0.1,0.1 -0.1,0 0,-0.1 z m -4.7,-1.3 0.2,0.1 0,0.1 -0.1,-0.1 z m 0.1,-0.5 -1.5,-0.3 -0.7,-1.5 -1.7,-0.5 0.8,-4 0.1,-4.5 1.4,-1.3 -0.7,-0.8 -4.2,-0.1 -0.9,-2.7 2.5,-1.4 1.3,0.4 7,0.1 3.4,0.3 3.4,1.7 2.8,0.2 0.2,0.4 0.3,-0.2 0.5,0.1 0,0 0,-0.1 0.1,0.1 0,0 1,0.3 1.3,-0.2 0.1,1 -2.3,1.6 -2,0.5 -2.5,3.5 1,1.6 -1.4,1 -0.4,1.6 -1.8,0.6 -0.8,1.2 -4.5,0.2 -1.7,1.2 z",
                    "BU" : "m 183.13,187.49 0,0 0,0 0,0 z",
                    "MV" : "m 476.83,206.89 0.5,0.7 -0.6,0 0.2,-0.4 z",
                    "SP" : "m 558.13,197.09 0,0 -0.1,0 0.1,0 z",
                    "GB" : "m 330.13,90.09 0.1,0.2 -0.2,0.1 -0.3,-0.3 z m 0.2,-0.8 0,0.5 0.3,0.1 -0.8,-0.1 z m 0.5,-0.2 -0.4,-0.2 0.8,0.3 -0.2,0.2 z m 3.5,-3.1 -0.5,1.4 -0.1,-1 -0.6,-0.3 z m 0.2,-0.9 0.3,0.1 -0.8,0.2 0.2,-0.5 z m 0.3,-0.4 -0.1,0.3 -0.3,0 0.1,-0.4 z m -154.3,95.9 0.1,-0.1 0.4,0.1 -0.3,0.1 z m 18.7,-4.9 -0.4,0 -0.1,-0.2 0.3,0.1 z m -1,-0.2 -0.1,0.2 -0.3,0 0.3,-0.1 z m 124.5,-81.9 0,-1.5 1.6,-0.9 -0.2,1.3 z m -100.7,234.1 1.4,-2.7 2,0.4 0.3,0.8 z m 179.2,-179.3 -0.1,0 -0.1,0 -0.1,-0.1 0,0 0.1,0.1 -0.2,0 0,-0.1 0,0 0,-0.1 0.1,0 0.3,0 0,0 -0.2,0 z m -0.3,-0.1 0,0 0,0 z m -74.8,-2.5 -0.1,0.1 0,-0.1 z m 73.4,3.3 -0.2,0 -0.3,0 0.1,-0.1 z m -77.1,-48.5 1.8,-0.6 1.5,1.7 -1.4,1.4 -2,0 -1.7,-0.9 z m 2.3,-6.2 2,-3.6 3.8,0 -2,2 1.1,0.6 3,0.1 0.3,0.6 -2.7,3.8 3,1.3 0.8,2.5 2.1,1.4 0.8,2.7 -0.6,0.5 3.2,0.3 -0.2,1.8 -1.7,1.5 1.3,0.9 -2.1,1 -7.1,0.3 -3.8,1 2.4,-2.5 2.1,0 -2.5,-1.5 -1.8,-0.2 2.1,-1 0.1,-2.5 1.9,-0.4 0.6,-2.1 -1.5,-0.9 0.5,-0.9 -2,-0.1 0.1,1.3 -1.4,-1.1 0.9,-1.8 -2,-2.4 -1.7,-0.7 1.5,-0.4 -1.9,-1.8 z",
                    "GR" : "m 386.63,147.99 -3,0.5 -2.3,-0.8 1.3,-0.6 z m 2.5,-2.3 1.2,-0.6 -0.3,0.8 -0.5,0.5 z m -0.8,-0.9 -0.4,-0.2 0.4,-0.4 -0.1,0.3 z m -0.5,-1 0.2,0.1 -0.2,0.2 0.1,-0.2 z m -3.3,0.2 0.7,-0.5 0.1,0.4 -0.4,0.8 z m 2.2,-1.4 -0.5,0.2 -0.2,-0.1 0.2,-0.1 z m 1.2,-0.3 0.2,0.1 -0.5,0.1 -0.4,-0.2 z m -1.7,-1.8 0.2,0.6 -0.4,-0.1 -0.2,-0.5 z m -1.2,-3.3 -0.4,0.1 0.4,-1.1 0.4,0 z m -8.5,-1.5 0.2,-0.1 -0.2,0.1 -0.1,-0.1 0,-0.1 0.1,-0.1 0.2,0 1.5,-0.6 2,-0.5 3,-0.6 1.4,0.8 2.1,-1.1 0.6,0.7 -1.2,1.5 -1.9,-0.6 -2.5,0.8 1.3,1.1 -3.4,-0.7 -0.1,1 5.4,5.8 -2.8,-1.9 0.3,1.3 -1.7,-0.6 -0.9,0.9 0.9,2.5 -2.5,-0.6 -1,-3.3 -2.6,-3.3 0.7,-0.3 z m 10.4,3.3 0.3,0.4 -0.3,0.2 -1.2,-0.5 1.1,-0.3 0,0 z",
                    "AS" : "m 9.63,245.09 -0.1,0 -0.1,0.2 -0.2,-0.1 z",
                    "DK" : "m 357.83,100.89 1,0.8 -0.6,0 -0.8,-0.3 z m -1.7,0.1 0.3,0.1 -0.2,0.1 -0.4,-0.4 z m 4.2,-0.3 -0.7,0.4 0,-0.3 0.3,-0.1 z m -5.3,-0.2 0.5,0.4 -0.1,0.2 -0.4,-0.1 z m 2,0.4 -0.2,0.5 -0.2,-0.2 0.7,-0.9 z m 3.4,-2 -1.2,2.7 -1.7,-2.6 2.2,-1.3 z m -6.1,2.4 -1.1,-0.3 -0.3,0 -1.1,-4.4 3.2,-1.5 1.6,-1.8 -0.7,2.3 1.4,1.3 -0.9,0.9 0.6,2.3 -2.4,-0.5 z",
                    "GL" : "m 234.93,60.89 0.2,0.2 -0.5,0.1 -0.2,-0.2 z m 1.5,-0.8 -1.8,0.4 0.2,-0.2 0.7,-0.2 z m -4.8,-5.7 -0.5,0 -0.2,-0.4 0.5,0.1 z m 2.2,-2.3 -0.9,-0.3 0.7,-0.7 0.5,0.7 z m -2.9,-4.6 -0.9,0.6 -0.2,-0.3 1.1,-0.5 z m -1.2,-1.6 -0.4,0.4 -0.7,-0.2 0.5,-0.2 z m 1.3,-0.1 -1.1,0.9 -0.1,-0.8 0.7,-0.2 z m -1.8,-2.8 0.9,0.5 -0.3,0 -1,-0.5 z m -0.7,-1.2 0.6,0.1 -1.2,0.1 0,-0.4 z m -0.4,-2.3 -1.1,0 -0.9,0 1.6,-0.3 z m -25.1,-8 -0.8,0 0.1,-0.2 0.8,-0.1 z m -5,-3.6 1.6,0.3 -0.6,0.2 -1.6,-0.3 z m 2.5,-0.2 1.3,0.4 -1.6,0 -0.4,-0.3 z m 102.6,-10.6 -2.2,1.3 -1.8,-0.7 3.2,-1.1 z m -94.9,-3 0.6,0.2 -0.4,0.1 -0.2,-0.1 z m 30,-6 -1.7,-0.1 -2.1,-0.5 0.8,-1 z m -4.8,50.1 0.6,-0.6 -3,-0.9 0.3,-1.7 3,0 2.4,1.4 -0.3,1.1 z m 1.4,12.1 -0.3,0 0,-0.2 0,-0.1 -0.2,-0.1 -0.7,-2.6 2.6,-1.4 -3.2,-0.5 1.5,-3.6 3.8,-1.2 0.7,-3.9 -1.2,-0.9 -4.6,-1.4 -1.4,-1.2 4.1,0.1 3.4,1.3 -1.5,-2.1 -2,-1.1 -0.3,0.2 -0.9,-0.1 0.3,-0.6 0.2,0 -0.3,-0.2 2.3,-0.6 -1.9,-0.2 -0.8,0.5 -0.5,-0.5 0.8,0 0.2,0 -2.2,-0.2 0.3,1 -2.6,0.2 -1.2,-1.1 0.6,-0.6 -0.3,-0.2 0.4,-0.1 0.1,0.1 1.2,-1.2 0.5,-2.1 -1,-0.4 0.1,-0.1 -0.8,-0.3 -0.1,0 -0.2,-0.1 0.8,-1.2 -1.5,-0.9 0.7,-0.9 -1.4,-0.7 0.5,-1.1 -1.8,-2.2 -2.1,-0.5 -0.9,-1.2 0.4,-1.4 -4.7,-1.7 -5,-0.8 -1,0.9 -4.2,-0.5 -0.5,1.3 -3.7,-0.5 -2.4,-1.3 3.2,-1.2 -4.6,-0.4 -1.8,-0.8 0.8,-0.8 4.9,-0.8 3,0.1 1,-1 -5.7,0.6 -0.5,-0.9 -3.5,-0.4 -3,-1.6 0.6,-1.5 6.2,-1.2 2.5,-1.2 4,-0.1 2.1,-1.7 -0.4,-2.3 -3.8,-0.1 -0.8,-1.3 4.5,-1.9 0.6,-1 3.5,-1 2.7,0.7 1.5,-2 -1.3,-1.1 5.5,-1.7 7.9,-1.2 1.7,1.2 -0.4,2.6 1.6,-2.3 3,0.7 3.7,-0.2 -2.2,-1 -1,-1.7 4,0.3 5.8,1.9 1.6,1.2 1.8,-0.3 -1.4,-0.9 0.4,-0.3 -4.1,-1.2 -2.8,-2 1.3,0.7 2.6,-0.2 3.6,1.3 -0.6,1.4 2,-1.4 -3.9,-1.8 9.1,0.1 -8.9,-0.5 -2.3,-0.5 1.5,-0.6 1.8,0.5 0.9,-0.9 4.8,-0.3 4.4,0.2 1.9,0.1 0.3,-1.2 3.7,-0.5 8.2,-0.5 3.5,0.2 9.6,1.4 -0.8,0.5 -9,0.3 -8,1.1 5.9,-0.4 3.4,-0.6 5.2,0.3 4.1,-0.5 1.4,1.3 3.5,0.4 2.5,0.9 -2.1,1.3 -14.2,1 -5.6,1.3 2.1,0.1 7.5,-1.1 6,0.4 -0.1,0.9 -3.1,0.8 -0.1,0.9 5,-1.5 1.1,-1.3 3.4,-0.3 0.3,2.6 -3.6,2.7 2.2,-0.4 6.4,-3.7 3,1.2 2.4,-1.2 3.6,0.1 3.6,0.8 1.3,0.8 -3.8,2 -2.3,0.1 0.5,1 -2.1,0.6 -3.2,-0.1 2.4,1.2 -1.2,0.8 -4.5,0.5 -3,0.9 2.2,0.8 -0.7,1.4 1.2,1 -1.8,0.6 0.9,1.3 -2,-0.6 -1.6,2.5 -0.2,1.6 4.2,-0.2 -0.3,0.8 2.9,0.4 -2.9,0.4 -2.8,-0.5 0.7,1.1 4.1,0.7 -0.8,2.8 0.4,2.3 -1.1,-2.1 0.7,-1.2 -3.4,-0.7 -2.3,1.2 0,1.6 3.4,-0.1 1.1,3.4 3.9,1.4 -2.1,0.2 -1.8,2.7 -3.6,0.7 1.3,2.4 -1.9,-0.1 -1.4,0.8 -3.3,-1.2 -4.1,1.3 1.1,0.3 -0.2,-0.1 2.3,-0.9 2.4,1.3 2.5,0.6 0.1,0.8 -2,-0.3 1.8,1 -1.3,0.1 0.9,1.2 -4.2,-1.7 -0.2,-0.8 1.5,0 -1.6,-0.3 -1.2,-0.5 -1,0.5 2.2,1.8 3.9,1.8 -1.1,0.7 2.4,1.6 0.7,2.5 -3.6,0.4 -1.8,-2.3 -3,-1.7 0.5,1.4 -4.5,1 -0.6,1.8 1.8,-0.2 -1.5,0 0.8,-1.1 4.5,-1.1 0,1.4 -1.7,0.7 -1.2,0 -2,1.3 1.1,0.3 3.8,-1.4 3.4,1 2.2,0 -1.6,1.6 -1.7,0.4 -4.1,2.8 -3.4,0.5 -2.6,1.2 -5,0.5 -2.6,1.6 0.2,0.8 -2.7,3 -3.3,1.7 -2.8,0.8 -0.3,-2.1 -1,1.8 -3.1,0.7 1.3,0.4 -1.5,1 -1.1,2 0.6,2 -3.1,3.4 -0.9,5 -1.6,1.2 0.4,1.5 -2.1,0.2 -2,-0.9 -0.2,-1.3 -4.7,-0.6 -2.1,-1.6 -0.5,-2.2 -1.9,-1 0.3,-1.4 -2.4,-1.6 0.2,-2.8 -1.5,0.8 0.1,-2.6 z",
                    "GU" : "m 613.53,191.99 -0.2,0.4 -0.1,-0.3 0.5,-0.5 z",
                    "MP" : "m 615.13,188.99 0.1,-0.4 0.3,-0.1 -0.2,0.3 z",
                    "PR" : "m 211.33,182.59 -1.8,0.7 -1.7,0 0,-1.2 z",
                    "VI" : "m 212.33,183.59 0.4,0 -0.7,0.2 0.1,-0.2 z",
                    "CA" : "m 175.63,27.69 -3.1,-0.2 0.9,-2.2 -2.3,-0.5 -2.3,0.3 0,-1.4 1.7,-1.5 2.6,-0.5 3.9,0.5 2.7,-0.3 -4.8,-0.3 -1.4,-2.9 -2.7,-0.6 0,-2.5 5.1,0.3 3.1,1.7 0.9,1.2 0.4,-1.6 -3.3,-1.5 9.9,-1.3 -3.7,-0.1 6.4,-1.3 -5.8,0.4 -6.3,1.3 -1.8,-1 -1,1.1 -4.5,-0.4 1.2,-1.8 -2.2,1.7 -4.1,-1.2 5.2,-0.5 0.3,-0.6 -6.3,0.5 -1,-0.7 3.7,-1.9 -4,1 1.8,-1.2 -4.6,0 1.8,-1 5.7,-1.2 6.3,0.6 -3.8,-0.7 2.7,-0.7 -0.8,-0.9 5.3,0.8 5.7,1.6 -5.2,-2 1.2,-1.4 2.9,0.6 -1.9,-1.1 3.2,0.1 -0.5,-0.8 5.8,0.3 1.9,-0.9 3.7,0.3 0.7,0.6 1.7,-1 6.3,-0.2 0.1,0.6 6.2,0.5 -4.3,1.2 2.3,-0.1 5.2,-1.2 3.4,1.9 2.6,0.1 0.5,1.4 -5.9,2.2 -1.9,-0.2 -2.4,1.2 -6.6,1.6 10.6,-1.8 -7.2,3.6 -4.5,2.7 -1.6,1.8 -3.6,-0.2 0.4,1.1 -5.3,0.8 -2.8,-0.5 0.1,0.8 5.5,0.2 0,0.8 -3.2,-0.4 -3.4,0.3 4.1,0.3 2.2,0.9 -0.3,0.8 -3.3,0.2 2.7,0.7 -0.9,1.1 -3.9,1.2 -1.3,1.8 -4.8,0.2 3.4,1 -0.7,0.7 2.8,-0.2 0.3,1.2 -2.9,1.4 0.8,1.8 -1.7,-0.1 0.8,-1.6 -3.3,0.7 0.5,-1.2 -6,-0.2 -0.9,0.6 -5.8,-0.5 -5.1,-0.2 -0.9,-0.9 4.3,-1.7 1.2,-0.9 -1,-1.8 1.9,-0.4 2.4,0.8 0.3,1.1 z m 0,0 0,0 0,0 0,0 0,0 0,0 1,-0.1 1.2,-2.1 -2.2,2.2 0,0 0,0 0,0 z m 9.3,102.5 0,0.2 0.3,0.2 -4.7,0.7 -2.4,1.4 -0.9,-0.7 1.2,0 -0.4,-0.5 0.3,-0.2 0.2,-0.9 1.3,-0.8 0.8,-4.4 2.3,1.8 0.7,-0.7 -1.9,-2.7 -6.5,-1 0,-0.5 -0.7,0.2 -0.7,-3.5 -1.9,-0.2 -1.1,-1.9 -3.1,-0.6 -2.9,2.5 -3.6,-0.1 -2.3,-1.2 -4.8,-1.2 -52.8,0 -0.4,-0.2 -0.2,0.2 -0.1,0 -0.4,-0.9 -2.7,-1.5 0.2,-0.7 -0.4,-0.1 0,0.1 0.3,0.2 -0.3,0.1 -0.2,-0.4 -0.5,-0.1 0,-0.1 -0.2,-0.2 -0.1,0.2 -0.6,-0.1 -0.1,0 -0.6,-0.1 0.5,0.1 0,0 -0.3,0 0.3,0.1 0.1,0 0.4,0.1 0.4,0.4 -0.1,0.1 0.3,0.3 0,0 1.2,1.6 2,1.6 -0.1,-0.5 0.5,0.3 0.5,0 -0.8,0.2 0.3,0.3 -0.5,0.7 -2.1,-0.6 -3.7,-2.1 -0.1,-0.5 0.1,-0.1 -0.3,-0.5 0,0 -0.1,-0.2 -1.9,-0.4 -1.2,-1.7 1,-0.3 2.6,0.8 0.1,-0.1 -0.2,-0.1 0.6,0 0.2,-0.2 -0.2,-0.2 -0.6,0.4 -2.3,-1.3 -0.3,-2.3 -0.3,0.8 -0.3,-0.3 0.2,-0.4 0.4,-0.1 0,0 0,-0.1 0.1,-0.2 -0.2,0.1 0,-0.1 0.2,0 -0.1,-0.3 -0.1,0.1 -0.2,-1.3 -1,-0.9 -0.5,1.2 -0.6,-0.9 0.1,-0.3 -0.3,0 0,-0.3 0.3,-0.2 0,0.4 0.1,-0.4 0.7,0.3 -0.2,-0.2 0.4,-2 -0.8,1 0.4,-0.1 -0.1,0.3 -0.4,0.2 -0.1,-0.2 -0.3,0.5 -2.2,-2.4 0.9,-1.9 0,-0.7 -0.1,-1.7 -3.4,-1.8 -3,-5 -3,-2.5 -1,-1.5 -1.7,0.6 -2,2 -0.4,-1 -3,-2.4 -3.5,-0.6 0,-29.2 3.6,0.5 3.6,1.9 2.6,0.1 0.1,-1.3 3.9,-0.7 1.1,0.3 5.4,-2.6 1.7,0.5 -4.7,1.7 -1.7,1.6 3.1,-1.8 0.6,1.1 1.1,-1.4 6.2,-2 0.9,1.8 2.3,1.4 1.8,-2.7 0.2,2.6 2.4,-0.5 0.4,-1.2 2.9,0.3 2.2,1.2 4.3,1.3 3.9,0.7 0.1,-0.6 3.4,1.5 0.2,1 -1.7,-0.1 -1.1,1.1 3.2,0.8 4.4,-0.1 2.3,-0.8 2.3,0.7 0.6,1.3 1.6,0.3 0.8,2.4 0.7,-1.9 -1.5,-1.9 4.5,-3.1 -3.8,1.6 -2.3,-0.4 1.1,-1.1 4.7,-0.9 0.9,1.8 1.6,1 2.2,0.1 3.5,1.1 1.6,-0.6 4.5,0.2 1.5,0.7 -2.2,-2.6 2.8,-0.3 0.8,1.4 1.8,-0.9 -1.1,2.7 2.2,1.9 -0.3,-3.8 1.2,0.1 2.4,-2.7 -2,0.3 0.6,-2.3 -3.6,-1.4 -0.7,-1.7 0.2,-3.3 2.4,-2.3 1.1,-0.2 -1,0.1 0,-1.8 -1.1,-1.2 0,-3.1 0.9,-1.4 3,-0.6 2.7,0.7 4,0.3 -3.2,3.9 -1,0.5 -3.4,-0.2 1.3,1.1 -1.2,1.6 -0.8,0.1 2.5,1.9 0.3,2.1 2.9,2.5 -2.5,1.5 2,0.7 2.6,2.1 0.3,2.1 1.2,-2.7 1.4,-0.6 1.9,1.7 -0.9,1.5 1.7,3.8 3.1,-3.1 0.5,-2.2 1.8,-1 -1.5,-0.8 0,-1.7 2.2,-0.3 0.9,0.2 0,-0.1 0.6,0.1 -0.1,0.1 1.9,0.4 2.4,1.5 -0.9,1 1.1,0.9 -2.7,0.6 2.7,3.3 -0.5,1.5 -3.5,2.1 -2,-2 1.3,2.5 -3,-0.3 -0.4,-0.9 -2.5,0.4 1.8,0.9 -2.9,2.6 -1.1,0 -3.3,-1.9 1.2,1.9 3.8,0.3 -1.8,3.2 -2,0.8 -1.8,-0.7 -0.4,1.6 -1.5,0.2 0,0.1 0.6,0.2 -0.1,0.1 -0.1,-0.2 -0.4,-0.1 0.6,1.5 -3.9,1.7 -3.2,4.8 -0.9,2.6 0.8,3.5 2.4,-0.1 1.4,3.6 -0.5,1.4 3.2,-0.9 4.1,1.1 2.5,2.2 5.7,2.2 4.4,0.5 0.3,3.3 -0.3,2.2 4.9,4.7 2.4,-3.1 -0.9,-1.8 -0.3,-2.9 -1.2,-1.3 3.8,-1.7 2.1,-2 0.3,-2.1 -1.1,-3.2 -2.8,-1.8 2.4,-3.4 -1.7,-3 1.2,-1.8 -1.2,-2.4 1.3,-0.9 5.2,1.3 2.1,-1 3.8,2.4 -0.2,0.9 4.6,1.7 -0.8,0.7 0.7,4.7 1.3,-0.1 2.1,2.3 3.6,-2.1 0.5,-2.5 1.3,-1.8 0.9,0.4 0.4,1.4 0.2,-0.4 0.2,0.3 -0.3,0.2 -0.1,-0.1 0.2,1 1.3,0.3 0.9,1.9 3,4.2 -1.1,1.3 4.1,4 4.8,1.7 -1.4,0.7 1.7,1.3 2.5,0.7 0.4,3.2 -2.6,2 -3.1,0.6 -2.4,2.2 -3.6,0.6 -1.2,-0.5 -7.8,0 -1.8,2.4 -3.2,1.4 -1.7,2.4 -2.7,2.5 -3.2,1.3 -3.3,2.9 -1,0.3 -1,0.9 -6.1,1.4 -1.6,1.3 1.4,0.1 0,0.2 z m -90.5,-17.9 0,0 0,0 z m 5.8,2.8 -0.1,-0.3 -0.2,0.1 0,0.1 z m -7.8,-7.2 0.3,-0.4 -1,0.4 -0.3,0.8 z m 57.5,-44.6 -0.4,0.1 0.1,0.3 0.3,-0.2 z m -54.8,52.4 0.3,0.1 0,-0.2 -0.1,-0.1 z m -0.2,-0.3 -0.1,-0.2 -0.2,0 0,0.2 z m 117.6,-39.5 0,2.5 -1.1,0.4 -2.5,-0.6 -1.9,-1.4 -0.2,1.1 3.2,2.7 -0.6,1 -4.4,-1.2 -2,-1.2 -4,-1.6 0.3,-1.2 -2.1,-0.8 -1.6,-2.4 -2.3,0.8 -2.2,-0.8 -1.7,1.3 -2.2,-0.5 -0.5,-1.8 1.3,-1.6 3.2,0.8 3,-1 1.4,0.2 -1.9,-2.2 2.9,-1.9 1.5,-1.7 -1.6,-3.2 -1.6,-0.3 -3.1,-2.3 -2.2,1.2 0,-1.2 1.9,-0.7 -4,-2 0,-1.2 -2.4,-0.5 0.3,1.5 -5.7,-0.9 -0.7,1.2 -7.9,-1.1 -4.6,-1.2 -1.7,-2.2 4.2,0.5 -1.4,-0.9 -3.3,-0.4 -0.5,-2.1 1,-3.2 1.2,-1.8 2.2,-1.6 2.1,-0.5 3,0.2 -3,3.6 0.5,3 3,2.6 -1.2,-3.6 1.9,0.6 -2.3,-1.9 1,-2.2 4.5,-1.9 2.5,0.1 2.4,3.6 -1.1,2.8 2,-1.9 1.7,1.7 0.9,-2 1.6,-0.6 4.6,0.9 -0.2,1.3 2,0.2 0,1.2 2.3,1.7 1.1,-1.4 2.6,1.4 -1.8,0.6 -0.1,0.4 0.6,-0.3 0.6,0.4 1.3,-0.6 -0.8,1.9 2.5,-1.1 3,1.1 2.4,2.9 -1.4,0.2 0.2,0.1 -0.4,0.3 0.4,0.1 0.4,-0.3 0.4,0.2 -0.5,0.2 0.7,0.1 0.8,1.1 -1.8,0.4 -0.7,0.8 -0.2,0 -0.4,-0.1 0.5,0.2 -0.4,0.5 2.8,0.6 0.6,0.8 0.2,-0.2 0.4,0 0,0.3 -0.6,-0.1 0.6,0.8 2.5,-0.4 2,2.5 0.2,0 0.4,-0.1 0.5,0.1 -0.1,0 0.8,0.1 -0.7,1.2 2.6,-0.4 1,2.1 -2.1,0.5 -0.1,1.7 -2,0.7 0.2,1.7 -1.9,-0.6 -4.4,-4.5 -2,0.2 1.5,1.5 -1.6,1.4 4.5,2.5 -0.2,0.8 2,1.6 0,0.2 0,0 0.1,0.4 0.3,0.3 z m -14.4,-23.2 -0.1,0.3 0.6,-0.2 -0.3,0.1 z m 8,4.1 -1.4,0.1 0.8,0.2 z m -5.8,64.5 -0.2,0 0.5,-0.4 0,0.2 z m -100.9,-5.5 0.3,0.4 0.4,0.2 -0.3,-0.1 z m 114.1,-2 4.3,1.4 -0.7,0.7 -2.6,-0.8 z m -116.5,-0.2 -0.2,-0.3 0.2,-0.1 0.2,0.3 z m -5.6,-3.7 -0.4,0 -0.1,-0.4 0.3,-0.2 z m 93.5,-1.6 -0.5,0.3 -0.6,0.2 0,-0.3 z m -99.5,0 -0.1,0.1 -0.1,0.2 0,-0.4 z m 5.4,0.2 -0.2,-0.2 0.2,-0.2 0.1,0 z m -0.1,-0.5 -0.1,0.2 -0.4,0 0.2,-0.2 z m -0.8,-0.7 0.1,0.4 -0.1,0.2 -0.2,-0.5 z m 0.3,-0.3 0.1,0.3 -0.1,0.2 -0.4,-0.4 z m -6,-0.3 0.4,0 0,0.2 -0.3,0.1 z m 6.3,0.9 -0.2,-0.6 0.2,-0.5 0.2,0.8 z m -1.8,-1.1 0.7,0.5 0,0.3 -0.3,-0.2 z m 2,0.5 0,-0.2 -0.1,-0.2 0.3,0 z m -2.5,-1.2 0.1,0.3 -0.2,0 -0.2,-0.4 z m 0.3,-0.2 0.2,0.4 -0.2,0 -0.2,-0.5 z m 91.9,-0.2 0.6,0.1 0.8,1.3 -2.6,-0.9 z m 3,0.2 0.1,0.1 -0.3,0.1 0.1,-0.3 z m -94,-0.5 -0.3,0.1 -0.1,-0.1 0.3,-0.5 z m -3.1,-0.3 0.1,-0.5 1.2,0.7 -0.1,0.6 z m 0.9,-0.4 -0.5,0 -0.1,-0.2 0.4,-0.2 z m 0.6,-0.1 0.9,0.8 -0.5,0.3 -1.4,-1.6 z m -1.7,-1.2 0.6,0.1 0.2,0.3 -0.7,0.4 z m -0.8,-1.3 0.7,0 0.1,0.6 -0.6,-0.2 z m 99.2,-0.8 -0.6,0.2 -0.1,-0.1 1.2,-0.2 z m -98,0.4 0,-0.1 0.5,-0.6 -0.1,0.5 z m 98.6,-3.5 0.4,-1 0.2,0 -0.1,0.7 z m -1.2,-1.5 0,0.5 -0.7,0.6 0.6,-0.6 z m -0.5,-0.7 -0.2,0.1 0.2,-0.3 0.2,0.3 z m 23.6,-3.9 -0.3,-0.1 0.1,-0.2 0.1,0 z m -2.9,-2 -0.6,0.1 0,-0.5 0.5,0 z m -21.7,-1.9 0.3,-0.2 0.1,0.2 -0.2,0.1 z m 0.7,-0.6 0.2,0.1 -0.2,0.2 -0.3,0.1 z m 29.3,-2 0.5,0.8 -0.5,-0.2 -0.2,-0.3 z m -6.5,0 0.5,0.3 -0.3,0.5 -0.8,0 z m -19.4,-0.8 -0.2,0.2 -0.7,0.2 0.2,-0.2 z m 25.7,-2.3 -0.1,0.9 -1.2,-0.8 0.8,-0.3 z m -1.9,-0.6 0.2,0.3 -0.5,-0.2 0.3,-0.2 z m -51.8,-0.3 -0.4,0.3 -0.3,-0.2 0.4,0 z m 53.2,0 0.5,0.5 -0.7,0 -0.2,-0.3 z m -27.5,-1.3 0.2,1.2 -0.8,1.2 -0.9,-1.7 z m -24.8,0 0.1,-0.1 0.4,0 -0.2,0.2 z m 53,-0.5 0.6,0 -0.4,0.5 -0.6,-0.1 z m -25.1,0 -0.5,-0.1 0.9,0 -0.3,0.1 z m 24.8,0 -0.5,0 0.2,-0.1 0.2,-0.1 z m -18.4,-0.5 0.8,0.1 0.2,0.2 -0.8,-0.1 z m 7,-0.4 1.2,0.8 -1.1,0.1 -0.9,-1 z m -21.6,-0.4 0.3,0.9 -2.3,1.5 -1.5,-0.9 1.2,-1.3 z m 29.3,0.1 0,0.1 -0.5,-0.5 0.2,0.1 z m -22.3,-1.6 1.4,0.1 0.3,0.8 -0.7,0.3 z m 27.3,-0.4 -0.2,0.4 0,0.5 -0.4,-0.9 z m -24.8,-0.2 1.1,0.6 -0.8,0.2 -0.9,-0.5 z m 24.7,-0.3 -0.2,0 0,-0.2 0.3,0.1 z m -25.5,-0.7 -0.4,0.2 -0.4,-0.1 0,-0.1 z m 24.1,-0.6 -0.1,0 -0.1,-0.4 0.3,0.2 z m -16.4,-0.9 -0.1,0.6 -0.1,-0.1 -0.1,-0.6 z m 15.8,-0.7 0.1,0.3 -0.8,0.4 0.4,-0.6 z m 5.4,-3.2 0.4,0 0.2,0.3 -0.6,-0.1 z m -39.9,-0.3 -0.4,0.1 -0.1,-0.2 0.4,0 z m -3.4,-0.7 0.6,0.6 0.1,1 -0.8,-0.7 z m 1.4,-0.2 1.1,0.5 0.7,1.1 -1.4,-0.5 z m 2.3,-0.6 0.4,0.1 -0.1,0.3 -0.6,-0.3 z m 39.7,-2.8 0.1,0 -0.9,0.5 0.3,-0.5 z m -86.1,0.3 -0.3,0 -0.2,-0.3 0.1,-0.1 z m -1,-1 -0.1,0.3 -0.3,-0.4 0.4,-0.2 z m -1,-0.5 0.2,0 -0.1,0.3 -0.3,-0.3 z m 85.4,0.1 -0.4,0 0.1,-0.4 0.3,0.3 z m -84.5,-0.1 -0.4,0 0.1,-0.4 0.2,0 z m -10.1,-1.2 -0.4,0 0.9,-0.1 0,0.1 z m 9.9,0.1 -0.4,0 0.1,-0.2 0.2,-0.1 z m -2.1,-0.3 0.5,0.1 -0.1,0.2 -0.4,-0.2 z m -0.4,-0.2 0.1,0.2 -0.3,-0.2 0,-0.2 z m -1,-0.4 0.5,-0.2 -0.6,0.4 -0.6,0.2 z m 68.2,0.3 1.6,0.3 0,0.7 -1.9,-0.1 z m -23.2,0.4 -0.4,0.7 -0.7,-0.6 0.4,-1.3 z m -48.4,-0.9 -0.2,0 0.1,-0.3 0.3,0 z m 69,-0.3 0.9,0.2 0.2,2.3 -1.6,1 -2.1,0 -0.7,-1.5 1.2,-1.8 z m -6.5,0.4 -0.3,-0.1 0,-0.4 0.7,0.2 z m -5.9,-0.6 0.2,0.1 -0.2,0.1 -0.3,-0.2 z m 15.2,0.4 -0.5,-0.6 0.3,0 0.3,0.3 z m -70.1,-1.1 0,0.1 0.1,0.2 -0.7,0 z m 11.9,0.6 -1,-0.5 0.8,-0.1 0.4,0.2 z m 49.7,-0.8 -0.1,0.5 -0.6,0 0.5,-0.4 z m 7.8,0.3 -0.5,0 -0.2,-0.3 0.2,-0.1 z m -0.7,0 0.1,0.7 -1.1,-0.6 0.3,-0.7 z m -51.5,-0.8 -0.1,0.7 -1,-0.4 0.6,-0.4 z m 3.1,-0.4 -0.5,0.6 -0.4,-0.1 0,-0.8 z m 0.3,-0.1 -0.4,-0.2 0.2,-0.5 0.2,0.2 z m -69,-0.6 0.1,0.1 -0.4,-0.1 0,-0.2 z m 65.8,0.1 0.1,-0.5 0.2,0.1 -0.1,0.3 z m 21.3,-0.9 0.2,0.4 -0.3,0.1 -0.1,-0.5 z m 23.2,-0.1 0,0.5 -0.8,1.1 -1.1,-0.1 z m -109.4,-0.7 0.3,-0.2 -0.3,0.7 -0.5,-0.2 z m 103.1,0.5 0.4,0.1 0.1,0.2 -0.8,-0.2 z m -16.2,0.2 -0.3,0.5 -0.3,-0.3 0.3,-0.3 z m 25.2,-0.1 0.7,0.1 -0.9,0.9 -0.5,-0.5 z m -46.4,-0.5 0.6,0.3 -0.5,0.3 -0.2,-0.2 z m 8.8,0 1,0 0,0.7 -0.8,-0.4 z m 1.9,-0.2 0.6,0.3 -0.7,0.7 0,-0.8 z m -82.7,0.1 -0.5,0.2 -0.2,-0.2 0.2,-0.2 z m 116.6,-0.3 -1.1,0.7 -0.6,-0.1 1.1,-0.8 z m -3.3,-0.5 0.3,0.7 -2.3,-0.2 1,-0.4 z m -5.4,0.2 -0.1,0.1 -0.5,-0.2 0.3,-0.1 z m -17.8,-0.2 0.1,-0.2 0.5,0.2 -0.1,0.3 z m -10.6,-0.3 -0.2,0 -0.1,-0.1 0.3,0 z m 19.8,-0.7 0.8,0 -0.4,0.5 -1.1,-0.5 z m -47.2,-0.7 -0.9,0.1 -0.3,-0.3 0.8,0.2 z m -9.2,-0.8 0.6,0 -0.9,0.2 -0.3,-0.2 z m -0.8,0 -0.7,0 -0.1,-0.1 0.1,-0.1 z m -21.5,-0.2 -0.3,0 -0.2,0.2 0.3,-0.4 z m 52.5,-0.3 0.4,0 0.5,0.8 -0.8,-0.3 z m 52.7,-2.6 0.7,-0.1 -0.4,0.5 -0.5,-0.2 z m 0.8,-0.4 -0.5,0 0.2,-0.4 0.3,0 z m -61,-6.4 -1.3,1.7 -2.3,-2.1 -0.2,-1.1 2.7,-0.1 z m 0.4,-11.5 0.7,-1.4 2.5,1.4 0.7,1.2 -2.6,0.5 z m 27,-5.3 1.2,1.2 -0.8,0.5 -2.1,-0.8 z m -28.2,0.9 1.1,1.1 -1.5,-0.1 -1.6,-2.3 z m 37.2,91.2 -0.2,0 -0.5,0 0.2,-0.2 z m -4.1,-2.7 0.2,0.2 -0.7,0 0.1,-0.3 z m 10.5,6.8 0.6,0.1 0.2,1.1 -2.7,-0.8 z m -2,0.2 -0.1,0.2 -0.3,-0.1 0.2,-0.2 z m -1.4,-0.8 0.5,0.2 -0.4,0.3 -0.2,-0.3 z m 14.7,5 -0.3,0.1 0.1,-0.2 0.5,0 z m -36.9,-101.2 4.6,0.2 -0.5,1.1 -4.8,-0.1 z m 68.6,96.9 -2.3,1.3 -1,-1.4 1.8,-2.4 0.3,1.8 z m -4.3,-3 0.1,1.9 -1.1,1.2 -1.3,-1.1 2.2,-0.2 z m -134.6,-14.3 -0.4,-0.5 1.4,-0.3 0.2,1.5 z m -1.3,-3.2 2.6,0.1 -0.7,2.3 -1.6,-0.5 z m 102.4,-5 -0.9,0.4 1.4,-2 0.4,0.6 z m -29,-63.2 2.3,1.8 -0.2,2.4 -2.9,-0.2 -2.9,-1.1 0.8,-1.8 z m -28.2,-10.5 -3.3,-0.2 -3.7,0.4 -0.3,-0.5 5.4,-1.5 1.7,0.7 z m 25.9,-0.7 2,0.6 -0.1,1 -3.8,1.4 -1.7,-1.1 -1.2,-1.9 0.5,-1.4 3.8,0.9 z m 55,101.8 -0.6,-0.2 0.1,-0.8 -0.5,-0.1 -0.3,-0.2 0,-3.2 -2.7,-0.9 -1.5,1.8 -0.5,2 -2.3,2 -6.2,0.1 2.3,-1 0.7,-1.5 5,-2.3 2.9,-3 5.4,-2.4 2.4,0 1.4,0.8 -0.2,1.1 -2.1,1.1 -2.1,-0.3 1.7,1.1 2.2,-0.5 -1.2,1.6 2,2.8 3.1,1 1,-0.6 1.8,1.3 -4,1.5 -1.5,0 -3.3,2.9 -1.2,-1.5 0.4,-1.1 2.2,-1.4 0.6,-1.8 -2.6,1.7 z m -12.4,-1 0.3,-0.2 -0.1,0.4 -0.1,0.2 -0.8,0 0.5,-0.3 -0.2,0.1 0.2,-0.3 0.3,-0.1 z m -96.9,-9.5 -0.8,-0.5 0.6,0.1 0.2,0.4 0,0 z m 7.2,-80.5 -0.7,-1.4 -0.9,1.1 -2.1,-0.1 -0.8,-1.2 2.5,-0.8 5.2,-3.7 4.6,-0.8 1.9,1 -1.6,0.5 0.7,2 -2.3,0.7 -0.7,1.7 -2.2,2.3 -1.5,-0.4 3.3,-2 -0.1,-2.6 -1.5,0.2 -1.1,2.5 z m 127.6,85 1.8,-1 0.5,1.4 -0.6,1.8 -2.4,-0.3 0.8,-1.2 -2.2,0.2 -1.5,1.1 0.1,-2 -6.7,0.3 -0.5,-0.8 1.4,-1.6 3.8,-6.9 2.1,0.9 -2.1,2.3 2.4,0.3 -0.4,1.4 3,-0.9 1.2,1.2 -0.8,1.7 1.6,0.3 z m -60.8,-43.2 -3,0.3 1.8,-1.5 -0.4,-1.6 0.9,-3.7 0.9,-0.5 1.3,2.2 0.4,-0.8 4.3,2.3 1.6,1.6 2.5,1.3 -1.5,1.1 -4.9,-2 -0.2,1 -3.1,2 z m -23.8,-20 2.2,1.3 2,2.1 1.2,0.1 -2.5,1.3 -1.8,-0.3 -4.1,-1.6 2.3,-1.1 z m -11.4,-3 4.7,2.2 1.2,1.1 -2.9,-0.1 -0.8,1.5 1.5,-0.3 0.3,1.4 -2.9,0.9 -5.8,-1.3 -0.1,-1.1 -1.6,1.6 -3.1,1 -8.2,0.9 -0.9,-2.5 -4.4,-0.3 -2.8,-2.4 0.9,-0.4 5.4,-0.7 5.1,0.2 -1.3,-0.8 -3.5,-0.8 -3.9,0.6 -3,-0.2 -1.5,-1.5 4.4,-1.2 0.5,-0.6 -4.5,0.6 0.2,-1 -1.9,0.1 0,-1 1.8,-1.1 -0.9,-0.8 2.4,-1.7 5.3,-1.8 1.2,0.9 0.4,2 1.4,-1.5 3.4,1.1 -0.8,1.5 2,-1.1 1.6,0.4 -1.8,-1.9 1.2,0.1 2.8,1.5 1.7,3.4 0.9,-1 -1,-0.8 -0.6,-2.3 0.2,-2.9 1.6,1.3 1.3,-0.1 2.1,1.6 0.2,1.2 1.8,3.2 -0.5,1.8 z m 49.5,-10.5 3.6,2.1 -0.1,1 -7.1,-0.1 -1.5,-1.6 -0.1,-1.7 z m -35.2,6.4 -3.1,0.9 -1.2,1.4 -3.6,-3.3 -2.3,-0.5 -1.8,-1.8 1.6,-1 1.2,1.4 1.7,-0.1 0.4,-1.4 -2.7,-1.4 1,-0.9 5.6,-1.3 0.8,-1.5 -0.5,2.1 1.5,0.3 0.2,1.7 -2.3,1.3 2.1,0.3 0.8,-0.8 1,2.7 z m -48,-9.7 2.7,1 4,0 2.3,0.7 3,2.3 -7.4,3.1 -1.9,1.7 -0.9,2.4 -4.7,1.4 -1.9,-2.1 -3.5,-1 2,-4 2.1,-2.6 -1.9,-2.2 z m 40.7,-1.8 -0.8,-1.6 1.8,-0.9 -5.4,0.6 -1.4,-1 2.2,-0.9 0.7,-1.9 3.9,2.1 -2.6,-2.5 0.4,-0.7 6.3,1 0.5,4 -1.2,1.8 z m -18.6,-3.4 1.4,-0.8 -2.1,-0.8 3.1,-2.3 0.5,3.3 2.8,0.6 0.9,-0.7 2.1,0.7 0,1.3 3.7,1.5 -2,0.5 -2,-1.2 -0.7,1.1 -4.6,0.6 -0.8,-0.6 -5.4,2.2 -3.9,0.2 -1.4,-1 3.1,-1.1 2,0 1.6,-1 -5.8,0.7 -0.2,-1.2 -1.7,1.6 -4.9,-0.9 3.1,-4.5 1.9,-0.7 0,-1 0.9,0.2 0.9,2.1 1.8,-0.3 3,1.7 0.4,1.2 4.5,0.2 0.1,-0.9 z m 58.7,3.9 -1.5,1 -3.1,0.6 -3.3,-0.4 -7.9,0.4 -2.3,-0.8 -2.1,0.5 -4,-0.9 -0.9,-1.7 0.7,-2.5 -1.9,-2 -1.8,0.4 -3.4,-0.5 -1.9,-2.4 2,-0.3 4,0.6 1.2,1.2 3.1,-0.3 4,2 -1.6,0.5 3.8,2.1 4.2,0 4.4,-1.2 5,0.2 2.8,1.2 0,1.2 -1.5,0.5 z m -59.1,-10.4 -3.6,0.8 -2.1,-0.9 -0.7,-1.2 -1.1,0.5 -1.8,-1.1 1.5,-0.5 2.1,0.8 4.8,-0.9 2,0.6 z m 12.5,-7.8 2.5,1.5 1.5,-0.3 4,2.1 -0.6,0.6 1.6,2.3 -4.2,0.2 0.3,-1.5 -3.3,0 0.1,-0.7 -3.3,0.4 -1.1,-0.9 2.8,0 -1,-2.1 -2.8,-0.1 0.3,-1.4 z m 24.9,-5.3 2.7,0.2 0.5,1.5 1.1,-1.1 1.5,0.1 1.4,2.2 -0.4,1.5 3.1,0 1.2,1.2 -3,1 -4.8,3.8 -5.8,-0.2 -4.3,-3.2 2,-0.9 2.6,0.1 -1.4,-1.3 -2.6,0.9 -3.5,-0.7 -1.8,-3.2 3.6,0.4 -3.3,-1 1,-1.3 1.6,-0.2 -0.6,-0.8 1.3,-1.2 3.2,-0.1 -2.4,-0.8 2.2,-0.5 2.8,1 z m -64.2,100.3 0.2,0.1 -0.1,0.1 -0.1,-0.2 0,0 z",
                    "ST" : "m 348.93,217.59 -0.3,-0.3 0.4,-0.3 0.2,0.3 z",
                    "CV" : "m 290.83,188.39 0.5,0.4 0,0.4 -0.5,-0.2 z",
                    "DM" : "m 218.73,187.89 0.3,0.3 -0.2,0.4 -0.2,-0.5 z",
                    "NL" : "m 346.93,104.89 -0.7,0.2 0.1,-0.2 0.8,-0.1 z m -131.3,78.2 -0.1,0 0,-0.1 z m 127.1,-72.9 0.8,0 0.9,0 -0.8,0.4 z m 7.4,-4.8 -0.4,2.6 -1.2,1 -0.7,2.7 -1.9,-1.8 -1.5,0.3 -0.8,-0.6 2.4,-3.5 2.6,-1.6 z",
                    "JM" : "m 188.03,182.19 2.4,1.1 -2,0.4 -2.2,-1.1 z",
                    "WS" : "m 6.43,243.59 0.1,0.6 -0.5,0 -0.5,-0.6 z",
                    "OM" : "m 444.03,168.69 -0.1,0.1 0,-0.2 0.3,0 z m -0.4,-1.7 0.6,-0.6 -0.2,1.5 -0.2,-0.1 z m -7.8,14.2 5.7,-2 1.3,-3.9 -0.9,-1.5 1.2,-4.4 1.1,-0.2 1.5,2.1 3.1,0.8 2,2.7 -4.1,5 0.3,1.3 -2.2,0.9 -0.6,1.3 -1.8,0.2 -0.8,1.6 -3.7,0.7 -0.7,-1.3 z",
                    "VC" : "m 219.13,192.79 -0.1,-0.3 0.2,-0.3 0,0.3 z",
                    "TR" : "m 386.03,136.89 -0.4,0.2 -0.2,-0.1 0.5,-0.3 z m 3.9,-4.1 2.1,1.7 -3,0.6 -0.7,0.8 -2.2,-0.3 1.2,-1.5 -0.6,-0.7 1.4,-0.8 z m 32.2,5.3 -1.6,0.6 0.5,3.1 1,1.8 -3.9,-0.4 -0.7,0.5 -0.3,-0.5 -4.7,1.3 -2.9,-0.3 -3.1,0.1 -1.4,2 0.2,-2.2 -1.3,0.8 -1.5,-0.4 -3.3,1.6 -1.5,-1.2 -2.6,-0.7 -0.5,1.4 -1.4,0.3 -2.6,-2.1 -2.2,-0.6 0.1,-1.4 -1.9,-0.7 1.6,-1.3 -1.2,-0.8 -0.1,-0.2 -0.5,-0.3 1.2,-2.1 4.6,-0.1 0.1,-1.8 3.9,0.3 4,-2.1 5,0.6 3,1.6 3.7,-0.1 1.3,0.4 2.7,-1.3 2.4,-0.2 1.2,1.1 0.5,2.2 2.1,1 z",
                    "BD" : "m 512.93,177.09 -1.5,-3.5 -2.5,2 -2.1,-0.6 -1,-3 0.4,-1.4 -1.4,-0.7 1.7,-1 -1.6,-1.5 0.7,-1.2 2.7,1 1,1.7 3.2,-0.1 0.8,0.5 -2.5,2.7 0.8,1.3 1.1,-1.5 0.8,3.5 0.1,1.3 z",
                    "LC" : "m 219.63,191.49 -0.3,-0.3 0.3,-0.4 0.1,0.3 z",
                    "NR" : "m 655.93,218.79 0,0 -0.1,-0.1 0.1,0 z",
                    "NO" : "m 376.13,55.29 -0.4,0.2 -0.4,-0.2 0.4,-0.4 z m -1.5,0.3 -0.5,-0.2 -0.4,-0.6 1.2,0.5 z m 5.1,-1.2 0.6,0.4 -1.1,-0.1 0.1,-0.3 z m 1.6,-0.8 0.3,0.5 -1.1,0.6 -0.5,-0.4 z m 0.6,-0.4 0.6,0.4 -0.8,0.3 -0.2,-0.6 z m -0.8,-0.4 -0.6,0.9 -1.8,0.3 0.7,-0.8 z m 1.3,-0.6 0,0.3 -0.3,-0.1 0.1,-0.3 z m 3,-0.5 0.6,0.5 -0.8,0.3 -0.5,-0.4 z m -8,-28.6 1.5,1.4 -2.3,0.2 -1.7,-1.2 z m 10.1,-0.2 -0.6,-0.3 0.3,-0.4 0.5,0.4 z m 4.4,-1.2 -1.8,0.4 -0.5,-0.1 1.1,-0.5 z m -16.9,-0.9 1.2,0.3 -0.9,0.3 -0.6,-0.2 z m -0.1,-1 -0.9,0.1 -0.1,-0.2 0.1,-0.1 z m 25.8,-3.7 -1.6,0.4 -2.3,0.2 -0.2,-0.2 z m -17.7,-0.8 -0.5,0.1 0.1,-0.3 0,-0.1 z m -0.4,-0.5 0.2,0.1 -0.2,0.1 -0.4,-0.1 z m -8.1,0.1 -0.2,-0.1 0.2,-0.1 0.3,0.1 z m 2.6,-0.9 -0.4,0.1 -0.2,-0.1 0.4,-0.1 z m -1,0.2 0,-0.3 -0.5,-0.3 0.8,0.2 z m -4.7,43 1.2,0 2.3,-1.3 0.6,0.6 0.2,-0.1 0.1,0 0.1,-0.2 0.3,-0.3 -0.2,-0.2 0.4,-0.1 0.1,0 1.1,-1 0.2,0 0,0.1 0.1,0.1 0,-0.1 2.3,0.3 3.2,-2 2.5,-0.3 -1.6,1.3 0.1,1.4 3,-3 -0.4,2 2.1,-2.5 4.6,1.3 2.1,1.5 -1.8,0.8 -2.7,-0.4 4,1.3 -1.7,1.5 -1.9,1.1 0.8,-1.5 -2.8,-2 -3.6,1.3 -0.8,2.7 -1.4,1.1 -1.8,-0.9 -2.9,0.4 -1.4,-1.9 -2,0.8 -1.3,2.3 -3.5,-0.6 -0.5,1.8 -1.1,-0.4 -2.2,2.2 0.5,1.2 -3.6,3 0,2.5 -1.7,2.3 0.6,1.8 -2.4,0.2 -1.4,2 0.7,3 -0.3,1.6 1.3,1.1 -1.2,1.1 0.5,2.5 -1.2,0.7 -0.8,2.5 -1.9,-0.7 -3.7,2.8 -1.8,0.6 -2,-0.2 -2.5,-2.9 0.8,-2.6 -1.9,-2.9 1.3,-1.5 -1,-0.8 1.5,-1.5 2.6,-1.1 -0.1,-0.6 3.6,-1.3 -1.9,-0.2 1.9,-1.2 2.5,1.5 2.6,-1.5 -2.9,0.9 -0.7,-0.8 3.2,-3.2 1.5,-0.8 0.3,-1.2 2.7,-4.7 3.8,-2.7 -1.5,-0.2 3.2,-1.3 -0.7,-0.6 2.2,-1.1 -0.8,-1.1 1,-1.6 1.8,-0.1 1.4,-1 -0.1,-1 0.9,1 z m 22.4,-0.9 -0.3,-0.3 -0.3,0.5 0.7,-0.2 z m -9.5,-24.6 0,-0.2 1,-0.7 -1,0.9 0,0 z m -20.5,28.4 2,-0.6 1.3,-1.5 0.9,2.1 -1.8,1.3 -4.3,0.8 0.2,-0.6 3.4,-0.7 z m 17,-31.5 -1.3,-0.8 -3.2,0.1 1.4,-2 -1.4,-0.7 3.9,-0.7 1.3,1.6 1.5,0.2 z m -14.5,-10.2 1.6,-1.6 4.5,2.7 0.1,1.2 5,1.3 -4.9,1.6 -1.5,4.1 -0.9,0.1 -1.7,3.6 -4.7,-2.5 -0.9,-1.4 5.5,-1.4 -5.9,0.4 -0.2,-1.2 2.8,-0.2 1.4,-1 -1.9,-1.6 -0.1,1.4 -3.3,0.8 -1.9,-1.7 0.3,1.7 -2.1,-1 -1.2,-1.9 1.7,-0.3 -1.2,-2.5 0.6,-0.6 2.1,0.1 3.2,-0.9 0.2,1.4 1.2,-1.1 1.5,0.9 2,2.9 z m 15.5,-2.7 5.8,0.6 0.7,1.4 -2.3,1.5 -3.7,1.4 -2.1,-0.2 -0.1,-0.8 -4,0.2 -2.2,-1.1 4.2,-0.4 0.1,-0.5 -6,0.5 0.1,-1.4 -1.9,-0.4 0.6,-0.7 2.3,0.8 2,-1.4 1,0.9 1.8,-0.3 1,1.2 0.7,-2.2 0.5,1.5 z m 2.5,16 -0.2,0.2 0.1,-0.2 0.1,0 0,0 z",
                    "KN" : "m 216.43,184.69 -0.5,-0.3 0.1,-0.1 0.2,0.2 z",
                    "BH" : "m 432.93,166.69 0.3,0.1 -0.1,0.8 -0.2,-0.4 z",
                    "TO" : "m 1.43,258.89 -0.1,0.1 -0.7,-0.3 0.4,-0.1 z",
                    "FI" : "m 375.83,86.59 -0.1,0.3 -0.6,0.1 0.3,-0.2 z m -0.8,0.1 0.1,-0.1 -0.2,0.4 -0.3,-0.2 z m 0.3,-0.5 -0.6,0.5 -0.9,-0.5 0.9,-0.3 z m 2.3,0.1 0,0.2 0.7,-0.1 -0.8,0.3 z m 1.6,0.7 0.1,-0.6 0.7,0.2 -0.6,0.7 z m -2.2,-1.7 0.1,0.3 -0.1,0.3 0,-0.4 z m 0,-8.1 0.2,0.3 -0.3,0.2 -0.2,-0.3 z m 6.9,-5.4 -0.5,0.3 -0.1,-0.2 0.2,-0.1 z m -5.9,13.9 -0.3,-0.1 -0.8,-4.4 0.5,-3.7 1.7,-0.3 4.6,-4.7 0.9,0.1 0.2,-2.2 -2.3,-0.9 -1,-2 0.7,-1.2 -1.1,-2.2 0.4,-1.6 -5.7,-3.5 2,-0.8 1.4,1.9 2.9,-0.4 1.8,0.9 1.4,-1.1 0.8,-2.7 3.6,-1.3 2.8,2 -0.8,1.5 -1,1.7 0.5,1 2.5,1.8 -1.7,2.7 1.5,2.3 -0.6,3.6 1.9,3 -1.1,0.8 3,2.5 -0.6,1.2 -3.9,3.8 -2.7,2 -7.7,1.6 -2.6,-0.9 0.1,-0.1 -0.3,0 0,0 -0.8,-0.2 0.2,0.1 -0.1,0.2 -0.3,-0.3 z",
                    "ID" : "m 593.33,228.59 1,0 -1.1,2.3 -0.1,-1.6 -1.9,-1.3 z m -51.2,-5.5 -1.9,0.6 -1.4,-1.8 0.5,-1.1 1.1,2.5 z m 19.2,10.6 2.9,-0.2 0,0.7 -3.9,0.9 -2.2,-0.5 1,-1.2 z m 20.3,-20.3 -0.2,1.9 1.6,2 -1.7,-0.4 -0.5,3.4 -1.2,-2 1.1,-0.1 -0.5,-2.7 z m -7.9,22.2 0.5,0.3 0.3,-0.6 0.3,-0.1 0.6,-0.3 0.1,0.4 0.2,0.6 -1.1,1.2 -1.9,0.5 z m -11.7,-25.7 -0.4,0.3 0,-0.4 z m 44.2,12.8 0,8.2 0,4.2 -1.8,-1.9 -4.6,0.6 0.7,-1.5 1.6,-0.6 -2,-3.9 -3.5,-1.5 -1.5,-0.1 -3,-1.8 -1.4,1.2 -1.3,-2.7 3.4,-0.6 -2.5,-0.1 -3.5,-1.9 -1.1,-2.3 1.9,1.3 2.8,-0.8 2.4,0.6 0.3,2.8 1.6,2.2 2.5,-1.5 -0.2,-1.7 3.2,-0.3 3.9,1.7 z m -39.3,14.7 -2.9,-1.4 1.4,-0.4 1,-2 4,0.6 1.3,-0.8 5.4,-0.4 -3.3,1.1 -0.5,-0.5 -3.9,1.1 -3.5,-0.1 1.7,2.2 z m -23.2,-4.7 -3.5,-0.8 -2,-1.2 1.1,-1.7 4.4,0.7 0.5,0.9 3.6,0.4 0.9,-1 2.2,0.9 6,-0.1 0,3.8 -3.8,-1 -1.1,0.3 -3.7,-0.5 -2.6,-0.9 z m 37.9,-9.1 -2.7,1.5 -1.4,-0.9 -0.1,-1.9 0.3,1.4 1.9,0.5 1.7,-0.9 3,-0.2 2,0.6 0.4,1.4 z m -10.2,5 -1.6,-0.4 1.8,-1.9 -2.7,0.3 0.2,-1.1 -2.3,-1.5 -0.2,4.3 -1.8,-0.3 0.5,-2 -1.6,-3.2 1.1,-1.3 1.3,-5.1 1.7,-1.2 1.3,0.5 4.4,0.5 2.3,-1.8 -1.2,2.5 -8,0.1 -0.6,1.2 1.3,2.2 1.9,-1.7 3.3,0.3 -0.2,2.1 -1.1,-1.2 -2.2,1.5 1.8,2.4 1,2.3 z m -10,-18.7 -0.4,1.6 1.4,1.9 -0.5,0.8 2.2,1.6 -2.4,0.6 -0.6,2.4 -1.6,1.7 0,1.4 -1.2,2.7 -2.6,1.1 -0.7,-1.5 -2.3,-0.4 -2.4,0.7 -0.1,-1.5 -2.8,0.5 -0.6,-3.6 -0.9,0.3 -1.2,-4 1.5,-2.4 1.7,2.3 4.7,-1.3 2.9,0.2 1.8,-3 0.8,-2.6 z m -37,-0.1 2.9,1.9 2.3,2.5 1.3,-0.5 4,3.6 -0.7,1.9 1.9,0.5 0.7,2.6 1.7,0 0.8,1.7 -0.4,4.5 -2.1,0.5 -6,-5.1 -1.5,-2 -1,-2.6 -1.8,-2 -1.8,0.5 1.6,3.3 -0.7,-0.1 -1.1,-3.2 1.5,-1.2 -1.1,-2.5 -2.3,-0.2 1.2,2 -0.5,0.8 -1,-2.9 -1.4,-0.5 1.7,0.3 0.5,-1.3 -4.1,-3.7 -0.3,-1.9 2.2,1.1 2,0 z",
                    "MU" : "m 446.83,256.49 -0.1,0.8 -0.6,0 0.3,-0.9 z",
                    "SE" : "m 371.43,89.89 0,0 0.1,-0.3 0.2,-0.1 z m 7.4,-19.2 0,-0.4 0.3,0.2 0,0.2 z m 3.7,-1.2 -4.4,0.9 -1.3,2.7 0.8,0.7 -1.5,1.7 -3.4,2.1 -2.3,2.1 -1.4,3.2 0.4,1.9 1.2,0.4 2.2,2.5 -1.2,1.7 -3.3,1.5 -0.6,2 0.6,2.7 0.7,-1.2 1.9,0.2 1.3,-1.8 -0.1,1.9 -3.1,0 -1.2,2.7 -3.4,0.3 -1,2 -2.3,0 -1.2,-4.1 -1.5,-2.9 0.2,0 0.2,-0.3 -0.5,0.1 0,0.2 -0.5,-1.1 0.4,-1.8 0.8,-2.5 1.2,-0.7 -0.5,-2.5 1.2,-1.1 -1.3,-1.1 0.3,-1.6 -0.7,-3 1.4,-2 2.4,-0.2 -0.6,-1.8 1.7,-2.3 0,-2.5 3.6,-3 -0.5,-1.2 2.2,-2.2 1.1,0.4 0.5,-1.8 3.5,0.6 1.3,-2.3 5.7,3.5 -0.4,1.6 1.1,2.2 -0.7,1.2 z m -23.7,22.4 -0.6,0.2 0.1,0.3 0.6,-0.2 z",
                    "TT" : "m 219.73,197.09 -0.2,1.3 -0.9,-0.3 -0.4,-0.8 z",
                    "MY" : "m 528.33,207.39 0,0.4 -0.2,-0.1 0,-0.3 z m 33.7,2.5 -0.4,-0.1 0.1,0 -0.3,0.1 -3.3,-0.5 -0.8,2.6 -1.8,3 -2.9,-0.2 -4.7,1.3 -1.7,-2.3 2.6,1 0.6,-1.6 0.2,0 0,-0.3 0,-0.2 3,-0.9 1.9,-2.8 1.2,1.1 0.7,-1.6 0.1,-0.1 0,0.2 0.6,1 -0.4,-1.2 1.9,-2.3 2.2,-1 3.8,2.8 -2.1,0.6 0.8,1 -1.4,0.3 z m -34,-4.4 1.8,0.4 0.1,1.2 1.8,-1.2 2.6,2.7 0,3.6 1.6,3 -1.7,-0.4 -4,-2.5 -1.1,-1.9 z",
                    "PA" : "m 187.13,203.99 -1.2,-2.6 -2,-0.7 -1.7,1.7 0.9,1.2 -3.4,0.4 0.1,-1.8 -2.2,0.3 -0.3,-0.6 0,-0.1 0,0 0,0 0,0 0.9,-2.3 0.9,1.2 2,0.3 2.7,-1.6 2.3,0.4 2,1.4 0.4,1.4 z",
                    "PW" : "m 593.83,203.79 0,-0.2 0.1,-0.3 0.2,0 z",
                    "TV" : "m 679.43,233.99 0,0 0,-0.1 0,0.1 z",
                    "MH" : "m 663.83,204.19 0.3,0.1 0.1,0 -0.1,0 z",
                    "CL" : "m 195.93,330.49 2.3,1.1 -1.4,0.9 -1.8,-1.9 z m -3.3,-14 1,2.1 0.1,3.2 -2.2,-0.8 1.3,-0.9 -1.3,-1 0.1,-1.9 z m 2.9,-12.8 -0.4,2.3 -2,0.7 1.1,-1.6 0.3,-2.5 z m 9.4,24.9 0,5.6 -0.2,0 0.2,0.1 -0.1,0.2 -2.5,-0.3 -1.1,1.1 -1.4,-2.5 1.6,-1.8 0.2,2 2.6,0.3 -2.1,-0.9 -0.6,-2.8 1.9,-1.5 z m -3.4,-75.6 0.8,-0.1 0.9,-1.5 1,2.8 1,1 -0.6,2 1.1,1.7 0.6,3 1.3,0 -0.3,2.4 -2.3,1.6 0.5,4.3 -1,0.5 -1.6,2.6 -1.8,6.6 1.6,3.8 -1.2,3.9 0,1.9 -1.2,1 -0.3,2.6 0.6,2.2 -1,0.5 -1.1,4.1 0.4,3.1 -0.8,2.1 1,2.1 -0.8,2 1.4,1.3 -1.1,4.6 -1.2,1.7 0.5,1 -2.5,3 0.8,3 1.6,-0.3 -0.2,2.3 1,1.1 3.7,0.1 2.9,0.9 -1.3,-0.4 -3,1.4 -0.6,2.7 -4.2,-1.4 0.5,-1.1 -2.2,-0.3 -0.6,-1.5 1.4,0.3 -0.3,-2.5 -2.4,-1.2 1.6,-0.9 -0.3,-4.5 -0.6,-2.2 0.9,-2.3 -2.2,-0.3 1.8,-2.9 0.9,1.4 1.5,-1.7 -0.3,-1.1 -2,-0.1 1.1,-2.4 0.5,2.3 2.1,-6.3 -0.9,-1.3 -1.5,0.6 -0.4,-1.8 1.3,-3.3 -0.8,-3.9 0.9,-1.3 1.1,-3.4 0.8,-1 1.6,-5.9 -0.5,-4.5 0.7,-0.6 -0.5,-2.1 1.2,-2.7 0.9,-4.6 -0.3,-4.6 1,-3.3 z m 6.1,81.7 1.3,0.8 -4,-0.3 3,2 -1.4,0 -2,-1.9 -1.9,0.6 -0.2,-1 4.7,-0.3 0.3,0 z",
                    "TH" : "m 532.13,194.49 0.1,0.1 0.2,0.3 -0.3,0 z m -0.4,11.4 -1.8,1.2 -0.1,-1.2 -1.8,-0.4 -1.6,-2.3 -2.1,-1.7 1,-3.5 1.7,-2.8 -0.9,-3.7 -1.7,-2.1 1.1,-2.6 -2.7,-4.6 1.2,-2.6 0.9,0.3 3,-1.3 0.7,1.6 1.5,0 -0.2,4 1.8,-1.5 1,0.8 1.5,-1.2 1.2,0.2 1.6,1.9 -0.2,1.6 1.7,1.6 -0.8,2.7 -3,-0.2 -2.5,1.7 1.1,3.7 -2.1,-2 -1.6,0.2 0,-1.7 -1.8,0.1 0,2.4 -1.4,3 -0.1,2.7 1.2,-0.2 1.1,4.1 2.1,0.6 z",
                    "GD" : "m 218.33,194.79 -0.2,0 0,-0.3 0.2,-0.1 z",
                    "EE" : "m 379.63,90.99 0.8,0.6 -1.6,0.6 -0.7,-1 z m 1.4,-0.1 -0.3,0.2 -0.3,-0.2 0.2,-0.2 z m -0.3,-1.2 0.3,0 0,0.2 -0.4,0 z m -0.9,0.1 0.6,0.4 -1,0.5 -0.9,-0.7 z m 9.5,0 -1.4,0.6 1.3,2.2 -0.2,0.5 -0.4,0.8 -1.6,0 -2.3,-1.5 -1.9,0.6 -1.5,-2 -0.1,-1.8 4.8,-1.2 3.9,0.4 0.2,0.2 0.1,0.2 -0.1,0 -0.2,0.1 0,0 0.1,0 0,0 0.1,0 -0.1,0 -0.2,0 -0.2,0.4 z",
                    "AG" : "m 218.03,184.89 0.2,0.2 -0.3,0 -0.1,-0.2 z",
                    "TW" : "m 565.23,172.19 1,0 1.9,-3.1 1.6,-0.1 -1,3.9 -1.1,2.5 z",
                    "BB" : "m 222.33,192.79 -0.2,-0.1 0.1,-0.4 0.3,0.3 z",
                    "IT" : "m 359.93,141.89 6.3,-0.8 -0.8,1.2 -0.2,2.4 -1.2,-0.3 z m -5.6,-7.3 0.8,1.5 -0.5,3.1 -2.2,0.4 0.1,-3 -0.5,-1.3 z m -3.7,-6 -1.7,-3.1 0.8,-1.9 2,-0.8 1.5,0.4 2.2,-0.7 0.9,-1.2 3.3,-0.5 0.4,0.9 2.5,0.4 -0.6,0.6 0.6,1.7 -0.9,-0.5 -2.1,1.1 0.5,2.6 2.3,1.6 0.9,2.2 2.1,1.5 1.7,0 -0.2,1 4,1.9 0.9,1.2 -2.8,-0.9 -1,1.9 1.2,0.6 -2.1,3.3 0.3,-1.9 -1.1,-2.9 -3.7,-2.6 -1.2,0 -3.6,-3 -2.1,-3.4 -2.5,-1 z m 9.5,4.4 0,0 0,0 0,0 z m 0,-4.6 0.1,-0.2 -0.1,0 -0.1,0.1 z",
                    "MT" : "m 363.63,146.09 0.5,0.2 0,0.3 -0.3,-0.1 z",
                    "VU" : "m 656.83,245.19 1.4,1.2 0.7,3.8 -1.7,-0.8 -1.6,-2.6 z",
                    "SG" : "m 535.13,214.99 0.3,0.2 -0.3,0.2 -0.4,-0.1 z",
                    "CY" : "m 400.93,148.29 0,0 0,0 z m 0.5,-0.1 0.2,0.2 -0.4,0 -0.2,-0.2 0.2,0 0.2,0 z m -0.5,0.1 0,0 0.1,0.1 0,0 z m -0.1,0.1 -0.2,0.3 -1.1,0.4 -0.4,-0.1 -0.1,0.1 -0.9,-0.9 0.6,-0.3 0.1,0.1 0,-0.1 0.1,0 0,0 0.1,0 0,0 0,0 0,0 1.3,0.1 0.5,0.3 z",
                    "LK" : "m 489.33,199.29 1.6,0.6 2,3.4 -0.1,1.9 -2.2,1.3 -1,-0.4 -0.6,-3.7 0.8,-2.7 z",
                    "KM" : "m 419.53,240.59 -0.5,-0.4 0.1,-0.7 0.3,0 z",
                    "FJ" : "m 675.43,250.99 2.2,0.1 0,-1 3.1,-1.2 -3.2,2.4 0.9,0.5 -0.4,2.4 -0.7,-1.3 -1.5,-0.3 z",
                    "RU" : "m 621.23,124.99 -2.1,0.7 -1.5,1.4 -2.9,1.4 1.3,-1.6 4.8,-2.3 z m -189,-65.8 -2.9,0.8 -0.5,-1.8 1.8,-0.8 1.7,0.8 z m 17.7,-4.7 2.2,1.5 -1.8,0.7 -1.9,-1.3 z m 22.6,-10.7 1,0.4 -3.4,0.7 0.1,-1.3 z m 80.8,-3.8 -1.1,1 -2.5,-0.9 1,-0.8 z m -107.6,-23.8 -2.8,0.4 0.6,-1 4.7,-0.6 0.7,2.5 z m -4.8,-1.3 -0.5,1.2 -3.3,0.2 0.5,-1.1 2.8,0.6 z m 2.9,-2.7 4.8,0.2 -3.3,1.7 -2.1,0.2 -3.5,-0.8 z m 16.4,-0.5 1.4,1.2 -4.5,1.1 -1.1,-0.7 2.9,-0.6 z m -13.1,-2.3 1,1 -1.4,1.6 -3.5,-0.2 z m -34.3,120.1 -1.9,-1.7 -4.6,-2.5 1.8,-0.4 0.6,-2.6 2.8,-1.3 -2.1,-0.3 1.2,-1.8 1.8,0.1 -0.1,-2.3 0.8,-1.2 -1.8,-1.5 -3.4,-1.4 -3.4,0.2 -1.1,-2.2 -1.7,-0.1 0.5,-1.3 -1.2,-1.5 -3.9,0.6 -1,-2.3 2.8,-1.1 -1.8,-1 -1.7,-4.8 -5.2,-1.4 -1,-1.9 -0.6,-1.9 0.4,-0.8 0.2,-0.5 0.4,-1.3 -0.3,-1.5 0.3,-0.5 0.2,-0.3 0.2,-0.1 0,0 -0.1,0 0,0 0,0 0.1,0 0.2,-0.1 -0.1,-0.2 -0.2,-0.2 0.2,-0.9 3.3,-1 -3.9,-1.1 2.7,-2 3.9,-3.8 0.6,-1.2 -3,-2.5 1.1,-0.8 -1.9,-3 0.6,-3.6 -1.5,-2.3 1.7,-2.7 -2.5,-1.8 -0.5,-1 1,-1.7 1.9,-1.1 1.7,-1.5 2.3,-0.6 2.1,0.8 -2.1,0.6 7.5,1.3 9.8,5 0.6,1.5 -1.8,2.6 -2.1,0.9 -2.7,0.1 -8.5,-2 -2.4,-1.3 5.2,4.5 -0.9,1.1 0.9,1.6 1.5,-1 2.4,0 4.9,1.9 1.7,-0.8 -1.2,-2.5 2.9,-1.4 2.4,-2.3 3.1,2.9 0.6,-2.6 -1.3,-1.6 0.8,-3.5 -1.8,-1.2 5,0.5 1.5,2 -2.6,0.4 -0.8,1.3 1.7,1.4 3.7,-0.3 0.5,-2.1 1.5,-0.3 4.5,-2.5 4.4,-1.4 1,1.9 1.4,0.5 2.5,-1.6 2.5,0.4 3,-1.5 1.5,2.2 2.4,-2.6 -1.5,-1.6 1.2,-0.9 6.5,1 2.9,1.6 3,0.9 2,1.6 1.6,-2 -1.2,-0.1 -0.9,-1.8 -2.3,-0.3 1,-3.9 -1.7,0.1 0.9,-2.1 3,-1.9 0.9,-3.1 1.2,-0.7 3.8,-0.1 2.4,0.7 0.1,1.6 -2,2.6 1.6,1.5 -0.4,2.9 0.2,4.6 2.1,1.4 -1,2.6 -1.1,0.4 -1.9,3.1 -2.3,1 3.3,0.3 2.7,-1.8 2.4,-2.8 -0.6,-3.4 3.7,-1 1.8,1.6 -0.4,2.5 1.9,-1.6 -0.8,-2.1 -4.2,-1.3 -3.3,0.2 -0.5,-1.8 1.6,-2.7 -2.5,-3.2 0.9,-1.4 2.7,-1 -0.4,-2.9 1.8,-2.4 2.1,0.3 -2.3,2.7 0.4,1.8 -1,3 4.9,0.9 0.9,-0.4 -3.7,-1 -0.6,-1.3 1.8,-0.5 1.2,0.8 0.6,-1.7 2.4,-2.9 0.5,2.6 4.5,2.5 2.9,-0.1 -1.9,1.5 1.1,1 0.7,2.4 1.1,-0.5 -1.2,-2.7 0.9,-1.4 -2.8,-2.3 -2.8,-1 -0.7,-2.4 0.4,-1.4 12.5,-1.1 -0.5,-1.1 -2.8,-2.2 3.1,0.5 -0.5,-1.3 2.1,-0.2 4.1,-2.2 6.9,-1.1 -1.2,-0.7 3.2,-0.1 2.3,-0.9 -0.9,-1.5 1.6,-1.3 0.5,2.6 -1.1,0.8 1.2,1.1 2.6,-0.5 2,-2.1 3.8,-0.1 0.6,-2.2 4,-2.3 2.8,-0.2 2.2,0.6 -1.3,1.5 4.3,1.2 -2,1.5 2.8,-0.1 0.1,-0.8 6.2,0 2.6,0.5 0.2,1.6 1.4,0 0.6,2.5 -2.1,-1.1 2,2.3 -7,3.6 -3.6,2.6 -1.8,0.1 -1.5,1.5 2.6,0.1 6.1,-1.9 -2.5,-0.1 0.8,-1.1 4.5,1.2 1.5,-0.7 0.6,1.4 3.7,-0.8 5.9,0.5 -0.2,1.1 3.6,1.1 5.2,0.4 1.1,-1.3 -0.5,-1.2 2.5,-1.3 0.3,0.7 3.2,1.1 1.2,-0.5 4.6,2 -1,3.4 -2,-1.3 3.6,4.6 2.2,1.4 1.2,-0.5 1.9,-3.5 2.2,1.6 1.6,0.2 2.8,-0.9 2,0.3 3.3,-0.3 2.5,0.6 -1.6,-2.5 0.6,-1.2 3.9,-1 7.4,0.8 2.9,0.9 -2,0.4 0.5,1.3 2,-1.8 4.6,0.6 0.9,0.9 -2,0.7 4.7,1.4 1.4,1.5 2.8,0.1 4.3,-0.8 5,0.4 2.2,1.1 0.8,1.5 -0.8,1.2 2.7,1.5 5.6,-1.1 1.6,0.7 4,0.2 2.5,-1.8 2.3,0.5 -0.5,1 -2.1,-0.4 0.5,1.6 2.4,1.5 2.6,-0.9 -0.7,-3.6 9.4,0.9 3.5,0.8 7.3,2.9 0.3,0.7 6.5,2.9 1.6,3.7 0.8,-1.4 4.4,0 3.9,2.7 -1.6,1.6 -3.1,0.3 -0.9,2.6 0.8,0.9 -2.2,0.3 -2.6,-1.6 -2.2,-0.6 -0.1,-1.2 -2,-0.7 -3,0.3 -2.2,-2.1 0.8,1.8 -0.8,1.5 -4,1.5 -3.5,-0.6 3.1,2 0.5,-0.5 2.6,5 -1,1.3 -2,-0.8 -2.6,0.2 -6.2,2.7 -2.8,2.1 -2.7,1.2 -0.4,1.4 -2.2,-1.8 -4.4,0.8 -1.6,1.3 0.6,-1.8 -3.1,2 -2.2,-0.8 -0.9,1.6 1.2,1.7 1.6,0.1 -2.2,1.2 -0.8,-1.5 -2.1,2.5 2.6,1 -1.1,2.6 1.2,1.6 -2.4,0.1 -0.7,1.4 0.7,2 -4,1.8 -0.8,2.4 -1.9,1 -1.7,3.4 -3.8,3.5 -1,-0.1 2.9,-1.9 -1.2,-5 -1.1,-6.2 0.9,-4 2.8,-2.9 2.9,-1.7 1.7,-2 3.8,-3 0.1,-0.6 4,-2.7 0.2,-2.7 2.1,-0.4 -1.2,-0.9 -2.5,0.5 -0.6,2.1 -5.4,3.6 -0.7,-1 1.1,-3 -5.5,0.5 -4.9,4.1 -1.6,2.5 2.1,0.9 -3.5,-0.2 -3.8,1.1 -1.2,-1.8 -2.2,-0.8 -1.6,1.5 -9.3,-0.5 -2.5,0.6 -3.2,2.3 -0.8,1.5 -3.5,2.4 -2.1,2.4 -4.5,3.2 2.5,0.8 0.3,-1 2.7,-0.1 -1.5,2.6 1.1,1.2 1.6,-1.9 2,0 3.2,3 -1.9,6.7 0.2,2.4 -0.7,2.7 -1.8,1.7 -2.2,3.7 -4.3,4.4 -1.4,2.1 -3.8,1.9 -2.6,-1.5 -2.1,2.4 -0.1,-0.3 -0.2,-0.3 0.9,-0.7 0.5,-2.8 -0.6,-1.9 2,-0.9 0.7,1.7 0.9,-1.3 2,-2.8 0.5,-2.4 1.2,-1.1 -0.7,-1.7 -3.7,1.7 -2.7,0.1 -0.4,-1.6 -2.8,-2.7 -2.5,-0.3 -4.5,-8.8 -4.7,-1.5 -4.7,0.8 -1.5,1.3 1.4,1.7 -1.3,1.1 -1.8,3.3 0.1,0.9 -2.8,1.2 -2,-0.7 -2.5,-0.2 -2.1,-1 -2.4,1.7 -4.4,1.2 -4.2,-0.5 -1.2,-1.5 -5,-1.4 -3.1,0.9 -2.7,-1.2 -0.2,-1.8 -6.3,-2.1 -2.1,2.9 1,1.2 -1.9,2 -5.2,-0.7 -0.7,-1.4 -3.6,-0.7 -8.7,4.2 -0.6,0.1 -0.3,0.1 -1.4,-1.3 -2.6,0.1 -3.5,-3.6 -3.8,0.7 -1.4,-1.4 -1.4,1.2 -4,-6.3 -2.6,-1.8 0.3,-1.3 -4.7,2.3 -1.4,-1 -4.5,-0.4 -0.6,-3.1 -4.9,0.2 -0.2,0.7 -5.7,1.6 -8.1,1.2 0.5,1.6 1.6,0.4 -2.7,1.1 0.7,0.9 -1.7,1.2 2.6,1.1 -0.2,1.7 -2.7,-0.2 -0.4,0.8 -2.2,-1.2 -4,-0.1 -1.7,1.3 -2.2,-0.8 -2.2,-1.7 -5.3,-0.3 -3.6,2.7 -0.5,1.9 -1.7,-1.6 -1.2,1.6 0.2,1.5 -1,2 1.2,1.4 1.8,0.1 2.2,3.5 -3,1.5 -1.8,2.5 1.7,2.1 -0.3,1.7 2.2,2.7 -1.6,1.5 -2.5,-1.6 -3.1,-1.9 -1.7,0.4 -2.1,-1.4 z m 271.5,-79 1.5,0.9 -0.9,0.7 -4,0.2 -2.3,0.6 0.2,-1.5 2.6,-1.2 z m -78.8,-8.9 2.7,0.1 2.6,1.4 -0.4,1.3 -5.1,-1.1 z m -92.9,-23.7 -1.9,-1.1 5,0.6 -2.1,1.1 z m -86.5,-4.2 2.6,-0.3 -4.2,1.7 -1.9,-0.9 z m 25.7,-0.9 2.6,-0.7 1.1,1.5 -2.3,1.7 -2.6,-0.1 -0.9,-1 z m -77.7,89.4 1.2,-0.6 -1.5,0.6 0,0 0.6,-1.2 1.9,-1 0.2,0 -0.9,0.9 1.4,-0.8 3,1 -0.1,1.4 -0.2,0 z m 144,-85.5 -1.4,0.2 -1,0.2 -4.2,-1.3 2.4,-1 0.5,-2.2 5.2,-1.3 4.2,2.6 -1.4,0.3 0,1.8 -3.6,0.6 4.3,-0.3 1.5,1.3 1.6,-0.8 1.4,1.6 -1,3.7 -3.2,0.1 -1.8,-0.8 -3.4,-0.2 -2.4,-2.2 z m 92.4,98.5 -1.3,3.3 1.6,3.6 -1.8,-0.2 -0.7,1.7 -0.5,-1.7 0.7,-3.3 -0.4,-2.1 0.5,-5.8 -1.1,-1.5 0.2,-4.2 2,-0.8 -0.5,-1.6 1.4,3 0.3,4.2 1.6,5.7 z m -166.2,-71 -2,3.4 1.4,3.7 2.9,1.9 -0.9,0.8 -3.3,-0.9 -0.5,0.7 -3.6,-1.1 0.6,-1.2 -1.3,-0.6 0.2,1.2 -2.9,-1.9 -0.1,-1.3 1.8,-0.5 1.4,-3.7 3.5,-1.2 z m 173.7,-7.9 2.4,-0.4 1.4,0.6 3.3,0.6 -0.5,0.9 -4.6,0.3 -4.1,-1.7 0.7,-1.3 z m -11.1,-3.4 2.3,1.3 2.7,0.1 2.7,1.3 -2.6,1.9 -2.9,-1.4 0.3,1.1 2,0.6 -2.6,0.4 -1,-0.6 -5.4,1.3 -2.1,-0.8 -2,-1.9 -1.6,-0.4 -1.1,0.3 0.4,-1.9 2.6,1 0.8,-1.4 2.8,-1 4,2.4 z m -141.2,-3.3 2.3,0.6 0.3,1.2 -3.5,1.9 -11,3 -4.3,3.4 -1.8,-0.5 0.1,1.4 -2.8,3.5 -3.4,-0.7 -1.5,0.5 -1.1,-1.7 2.2,-0.8 1.4,-2.5 2.9,-2.9 2.2,-0.7 1,-1.2 4.5,-0.8 0,-0.7 2.5,0.3 3.2,-0.5 z m 63.7,-7.4 0.3,-1.4 2.4,-1.3 1.6,0.5 -0.4,1.2 1.7,-0.5 2.4,1.3 2,2.5 -1.7,-0.9 -2.9,0.9 -4.7,0.3 -2.3,0.9 -1.5,-0.3 z m -103.8,-6.5 5.7,-1.5 -1.2,-0.6 2.6,-1.1 2.6,1.5 -3.8,0.9 0.9,2.2 -1.3,-0.7 z",
                    "VA" : "m 360.13,132.99 0,0 0,0 0,0 z",
                    "SM" : "m 360.13,128.39 -0.1,-0.1 0.1,-0.1 0.1,0 z",
                    "KZ" : "m 451.63,123.39 -2.1,0.7 0.6,1.3 -1.4,-0.8 0.4,-0.3 -0.6,0.2 -5.1,1.3 0.1,8.5 -0.9,0.1 -2.6,-2.4 -1.3,0.5 0,0 0,0 0,0 -0.8,0.3 0,0 0,0 0,0 -1.2,0.5 0.5,-2.2 -1.5,-0.3 -3.1,-4.2 2.1,-1.6 2.1,-0.2 1.1,-1.4 0,-2.1 -1.8,0.1 -1.8,-0.7 -3.8,1.9 -2.2,-3.5 -1.8,-0.1 -1.2,-1.4 1,-2 -0.2,-1.5 1.2,-1.6 1.7,1.6 0.5,-1.9 3.6,-2.7 5.3,0.3 2.2,1.7 2.2,0.8 1.7,-1.3 4,0.1 2.2,1.2 0.4,-0.8 2.7,0.2 0.2,-1.7 -2.6,-1.1 1.7,-1.2 -0.7,-0.9 2.7,-1.1 -1.6,-0.4 -0.5,-1.6 8.1,-1.2 5.7,-1.6 0.2,-0.7 4.9,-0.2 0.6,3.1 4.5,0.4 1.4,1 4.7,-2.3 -0.3,1.3 2.6,1.8 4,6.3 1.4,-1.2 1.4,1.4 3.8,-0.7 3.5,3.6 2.6,-0.1 1.4,1.3 -3,1.7 -0.5,3.3 -1.5,0.5 -3.3,-0.9 -1.4,4 -4.2,1.2 1.4,4.4 -1.2,2.2 -3.4,-1.6 -6.2,0.1 -1.9,-1 -1.3,2 -3.2,-1 -1.7,1.4 -3.6,2 -0.9,1.7 -1.3,-1.3 -2.3,0 -0.4,-1.8 -0.9,-0.1 -0.5,-2 -1.6,-1.9 -5.6,0.5 -1.9,-2.1 -1.8,-1.1 -0.7,-2 1.3,-0.5 z m 0.3,0 0.1,0 0,0.1 1.5,-0.7 z m -19.5,2.7 -0.4,-0.2 0.1,-0.3 0.6,0 z",
                    "AZ" : "m 422.53,134.89 -0.1,0.1 0,-0.1 0.1,0 z m 0.4,0.2 -0.1,0 0,-0.1 0,0 z m 0.5,0.8 0,-0.1 0.1,0 -0.1,0.1 z m -1.3,2.2 -0.1,-0.1 1.9,0.3 0.7,1.5 -1.3,-0.3 z m 3.1,-5.1 2.5,1.6 1.6,-1.5 1.9,2.8 -1.3,4.9 -1.7,-0.9 -0.1,-2 -2.8,1.9 -1.6,-2.6 0.6,-0.4 -1.9,-2.4 2.9,0.5 z",
                    "TJ" : "m 471.53,135.09 0,0.1 -0.1,-0.2 0,0 z m 0,2.5 0.1,0.1 -0.2,0.1 -0.2,-0.2 z m 5.8,0.9 0.3,2 1.8,0.1 0.3,2.8 -3.1,-0.5 -3.2,1.7 0,-2.6 -2,-1 -0.9,1.8 -1.6,0.9 -2.9,-0.2 1.1,-2.3 -0.4,-1.6 -1.5,-0.6 2.1,-0.7 1.9,-2.8 2,-0.5 0.9,1.8 -1.9,0 -1.2,1.6 4.1,-0.2 1.5,0.9 z",
                    "LS" : "m 390.23,273.89 0.9,-0.3 1.5,1.6 -2.6,2.8 -2,-2.2 z",
                    "UZ" : "m 473.73,137.49 -0.2,0 0,-0.1 0.2,-0.1 z m -1.2,-0.3 0.1,0.3 -0.3,0.1 -0.2,-0.5 z m -22.4,-11.8 -0.5,1 1.4,1.2 0.3,-1.5 1.8,1.1 1.9,2.1 5.6,-0.5 1.6,1.9 0.5,2 0.9,0.1 0.4,1.8 2.3,0 1.3,1.3 0.9,-1.7 3.6,-2 -1.5,1.5 2.4,1 0.6,-0.7 2.7,1.5 -2.8,1.5 -1.4,-0.2 -0.9,-1.8 -2,0.5 -1.9,2.8 -2.1,0.7 1.5,0.6 0.4,1.6 -1.1,2.3 -1,0 -1.4,-0.3 0.1,-1.5 -1.8,-0.5 -6.1,-3.9 -0.9,-2.3 -3.6,-0.9 -0.4,-2.1 -2.5,-1.1 -1.8,1.5 -0.4,-0.4 -0.7,0.9 -0.1,1.5 -1.9,-0.1 -0.1,-8.5 5.1,-1.3 -0.6,2.6 0.8,-2.5 z m 21.4,9.7 -0.1,-0.1 0,0 0.1,0.2 z",
                    "MA" : "m 325.92969,146.28906 -1,0.30078 -1.69922,3.80078 -3.30078,1.59961 -1.40039,1.5 -1.09961,2.5 0.5,2.09961 -1.90039,2.90039 -1.69922,1.40039 -2.80078,0.79883 L 311.125,164 h 8.54883 l 0.0566,-0.21094 v -2.09961 l 1.90039,-1.5 3.69922,-0.5 4,-2.29883 v -1.90039 l 2.19922,-1.09961 2.40039,0.0996 0.40039,-0.90039 -1.19922,-1.5 -0.20117,-3.19922 -0.90039,-0.80078 -1.19922,-0.0996 -0.0996,-0.30078 h -0.0996 v -0.0996 l -2.70117,0.40039 -1.79883,-1.5 -0.10156,-0.0996 z",
                    "CO" : "m 185.43,214.99 0.2,-1.4 1.6,-1 1.4,-2.2 -0.4,-1.1 0,-4 -1.1,-1.3 1.4,-1.4 -0.4,-1.4 0.9,0.3 2.5,-1.8 0.2,-2.1 2.7,-1.5 1.6,0.1 3.4,-2.2 0.3,1.1 -1.2,0.4 -1.8,2.4 -0.2,2.1 1.2,1.8 0.5,2.5 3.8,0.2 1.3,1.6 3.8,-0.1 -0.8,3.1 1.1,2.2 -1,1 1.2,0.9 0.6,2.2 -0.9,-1.6 -1.5,0.7 -3.3,0 -0.4,3.6 1.2,2.4 -1,5.3 -1.5,-0.8 1.4,-2.1 -1.9,-1 -2.6,0.5 -1.6,-0.4 -0.8,-1.8 -3.2,-2.2 -1.9,-1.1 -2.2,-0.4 z",
                    "TL" : "m 573.73,235.59 0.4,-0.2 0.4,-0.1 -0.3,0.6 z m 1.7,-0.7 0.5,-0.7 3.5,-0.6 -1,1.2 -2.7,1.1 -0.2,-0.6 z",
                    "TZ" : "m 403.23,239.89 0,-0.2 -0.1,-0.1 0,-0.1 -0.3,-0.1 0,-0.1 0,0 0,-0.1 -0.5,-2.5 -0.4,-0.4 -0.2,-0.2 0,0 -0.2,-0.2 0,0 0,0 -0.2,0.1 -0.1,0.3 -1,-0.2 -0.9,-0.4 -1.8,-0.6 -1.6,-1 -2.3,-5.1 -0.5,-2.8 2.2,-2.3 -0.5,-1.6 0.5,-1.5 -0.7,-1 0.7,-0.1 1.8,0 0,0.1 0.1,-0.1 0,0 0.2,2.7 2.5,0.1 1.6,-2.7 6.9,3.8 0.2,1.1 2.7,2 1.3,1.1 -1.1,0.9 0.5,1.3 -0.4,4.4 2.1,3.4 -5.7,2.4 z",
                    "AR" : "m 204.93,334.29 0.1,0.2 -0.2,0 z m 26.8,-66.8 1.4,0.1 0.1,3.1 -3.7,2.2 -3.5,4.1 -1.2,3.6 0,1.2 -0.7,3.9 2.8,2.5 -0.5,0.9 1.4,2.2 -1.8,2.8 -2.8,1.2 -3.5,0.7 -2.3,-0.2 -0.5,4.4 -2.8,0.7 -2.1,-1.1 -0.3,2.9 0.9,1.1 1.6,-0.8 0.3,1.5 -1.6,-0.5 0,1 -1.7,1.6 -0.5,3.2 -1.2,0 -2.1,1.4 -0.6,1.3 1.6,2.1 1.6,0.1 0,2.4 -3.1,2.4 -0.6,2.4 -1.9,0.8 -0.6,1.7 1.4,3.5 -2.9,-0.9 -3.7,-0.1 -1,-1.1 0.2,-2.3 -1.6,0.3 -0.8,-3 2.5,-3 -0.5,-1 1.2,-1.7 1.1,-4.6 -1.4,-1.3 0.8,-2 -1,-2.1 0.8,-2.1 -0.4,-3.1 1.1,-4.1 1,-0.5 -0.6,-2.2 0.3,-2.6 1.2,-1 0,-1.9 1.2,-3.9 -1.6,-3.8 1.8,-6.6 1.6,-2.6 1,-0.5 -0.5,-4.3 2.3,-1.6 0.3,-2.4 1.9,-2 3.1,0.8 0.5,1.3 0.8,-1.7 2.4,0.5 3.2,3.1 1.8,0.4 4.8,2.9 -2,3.9 5.4,0.2 1.9,-1.6 z m -26.8,66.7 0,-5.6 0.3,1.7 2.2,2 2.8,1.6 2.4,0.6 -3.7,0.4 -1.3,-0.2 -0.5,-0.1 z",
                    "SA" : "m 429.03,161.99 0.9,1.9 2.4,1.9 0,1.9 1.3,2 0.3,0.4 0.4,-0.1 0.2,0.6 0.5,0.1 1.9,2.7 5,0.4 0.9,1.5 -1.3,3.9 -5.7,2 -5.5,0.8 -1.8,0.9 -1.4,2 -4.4,-0.6 -3.8,0.2 -0.7,1.9 -3,-4.6 -0.5,-1.6 -2.5,-1.9 -1.1,-1.7 0,-2.5 -1.2,-2.4 -1.8,-1 -0.5,-1.9 -3.9,-5.8 -1.2,-0.2 0.7,-2.5 2.1,0.3 3.7,-2.7 -2,-2.1 4.2,-1.3 2.5,0.4 3.1,1.8 5,3.9 3.6,0.2 1.7,0.2 z",
                    "PK" : "m 479.03,143.89 2.7,1.2 1.6,1.8 0.5,1.2 -2.4,1.3 -2.8,-0.6 -1.1,0.9 0.5,2.5 2.4,2.2 -1.5,0.9 0.1,1.3 -2.2,2.5 -0.9,1.8 -2,2.3 -2.7,-0.2 -1.9,2.5 3,4.3 -2.1,1.1 -2.1,-0.2 -1.3,0.8 -1.6,-0.4 -1.9,-2.8 -5.4,0.5 -3.7,0 0.5,-2.1 2.4,-0.8 -0.7,-3.3 -1.6,-0.7 -2,-2.7 3.1,1 3.1,0 4,-0.9 0.3,-2.3 3.4,-1.9 2.3,-0.2 0.4,-2.5 1.5,-0.5 -0.6,-1.5 2,-0.1 1.1,-2.4 -0.9,-1.9 2.7,-1.7 z",
                    "YE" : "m 435.83,181.19 1.4,3.3 0.7,1.3 -1.6,2 -5.1,1.7 -1.7,1.4 -1.3,0 -2.5,1.2 -1.7,0.1 -3.6,1.5 -0.9,-0.5 -1.4,-5.3 0.1,-1.5 0.7,-1.9 3.8,-0.2 4.4,0.6 1.4,-2 1.8,-0.9 z",
                    "AE" : "m 444.03,167.89 0.2,0.2 0,1.1 -1.1,0.2 -1.2,4.4 -5,-0.4 -1.9,-2.7 4.9,0.2 3.7,-3.9 0.2,0.8 z m 0,0.8 0.2,-0.1 -0.3,0 0,0.2 z",
                    "KE" : "m 411.33,226.69 -2.7,-2 -0.2,-1.1 -6.9,-3.8 0.3,-1.6 -0.4,-0.9 1.8,-2.7 -0.1,-1.5 -1.8,-3.4 2.8,-1.5 1,0.8 1.7,0.3 2.4,1.6 2.8,0.4 2.3,-1.7 2.2,0.6 -1.8,2.2 0,7 1.1,1.6 -2.6,2 z",
                    "PE" : "m 192.13,217.99 3.2,2.2 0.8,1.8 1.6,0.4 2.6,-0.5 1.9,1 -1.4,2.1 1.5,0.8 -1.5,-0.1 -4.1,1.8 -0.4,2.4 -1.7,2.2 3.5,4.8 1.9,-0.2 1.3,-0.9 -0.3,3 2.1,-0.1 1.7,3 -0.8,2.2 0.4,1 -0.9,2.6 -1.3,0.6 1.8,0.8 0.1,0 0.1,0 -0.2,0.5 0.1,0.1 -1.2,1.5 0.3,0.4 -0.9,1.5 -0.8,0.1 -2.1,-2 -7.1,-3.8 -2.1,-2.5 0.1,-1.4 -2.7,-4 -2.5,-5.9 -1.9,-2.7 -2.2,-1.4 -0.3,-3.5 1.8,-1.6 -0.4,1.3 2.8,1.8 1.4,-3.1 3.2,-1.5 2.1,-2 z",
                    "DO" : "m 198.63,182.19 0,0 -0.1,-0.1 0,0 -0.1,-0.1 0.6,-0.3 -0.1,-1.9 1.4,-0.5 2.1,0.6 0.6,1.1 2.4,0.9 -1.1,0.4 -3.4,0 -1.4,1.6 -0.6,-0.8 0.1,-0.6 z",
                    "HT" : "m 198.63,182.19 0,0 -0.1,-0.1 0,0 z m 0.3,-2.4 0.1,1.9 -0.6,0.3 0.1,0.2 0.1,0 0.4,0.3 -0.1,0.6 -3,-0.4 -1.1,0.4 -0.7,-1.3 3,0.5 -0.1,-2 -1.3,-0.7 z",
                    "PG" : "m 606.23,235.09 0,-4.2 0,-8.2 5.8,2.3 1,0 2.4,2 -0.1,1.2 3.3,0.9 0.8,1.4 -1.8,0.1 0.5,1.4 1.8,1.1 2.1,2.8 3.3,0.2 -0.5,1.8 -5.2,-0.7 -1.7,-1.4 -1.9,-2.6 -4.5,-1 -0.7,2.7 -1.4,0.7 z m 17.2,-12.6 -0.6,-1.7 1.1,1.5 3.5,1.6 2.1,2 -0.5,1.1 -1.2,-2.4 z m -0.6,7.3 -3,-1.1 5.6,-0.6 1.2,-0.9 -0.2,-1.5 1.7,0.3 -0.5,2 -3.2,1.7 z",
                    "AO" : "m 359.73,228.69 0,-0.5 -0.4,-0.9 1.4,-1.2 0.6,0.5 -1.2,0.8 z m 21.3,23 -4.9,0.7 -3.9,-0.5 -0.6,-0.7 -8.6,0 -1.5,-0.9 -2.7,0.6 0,-2.8 1.4,-4.6 2.2,-2.3 0.2,-3 -1.4,-3.1 0.7,-1.4 -2.1,-4.3 1.7,-0.5 6.6,0.1 0.7,2.7 1.2,1.5 3.3,-0.2 0.4,-1.9 4.3,0.5 0,4.1 0.7,1 0.2,2.6 3.3,-0.8 0,4.1 -3.8,0 0,6.2 z",
                    "KH" : "m 542.13,189.59 0,4.6 -3.4,2.5 -2.4,1.2 -1.6,-0.2 -1.4,-2.2 -1.1,-3.7 2.5,-1.7 3,0.2 3.1,0.1 z",
                    "VN" : "m 536.33,197.89 2.4,-1.2 3.4,-2.5 0,-4.6 0.3,-1.1 -2.2,-3.4 -3.4,-3.5 -1.7,-1 2.1,-1.5 -1.7,-1.8 -1.8,0.2 -0.3,-1.4 -1.6,-1.7 0.7,-0.7 2.8,0.5 2.6,-1.7 2.8,1.2 -0.4,0.8 2.7,1.8 -2,1.1 -1.7,1.8 -0.8,2 1.9,3 4.1,4.1 1.4,4.8 -0.9,3 -3.3,1.9 -1.1,-0.3 -0.4,1.9 -3.3,1.4 0,-2.7 z",
                    "MZ" : "m 403.03,243.59 -0.3,-2.5 0.5,-1.2 4.8,0.3 5.7,-2.4 0.8,8.3 -1.8,3.2 -5.4,2.8 -4.2,4 -0.5,1.1 1.6,3.3 -0.7,5 -2.6,1 -2.4,1.8 0.8,1.8 -1.1,0 -0.4,0 -0.2,-1.2 -0.2,-0.6 0.1,-3.1 -1.3,-4.1 2.1,-2.2 1.2,-2.5 -0.5,-1.1 0.5,-2.2 -0.1,-3.3 -4.9,-2 -0.4,-1.3 5.7,-1.9 0.9,1.1 1.3,-0.4 -0.2,2.9 1.7,1.3 1.3,-1 0.1,-2.7 z",
                    "CR" : "m 176.13,196.89 0.4,0.8 1.7,1.8 -0.9,2.3 0,0 -0.3,-0.5 -5.1,-3.3 0.3,-1.4 2.6,0.2 z m 1.5,5.6 -0.3,-0.6 0,-0.1 0,0 z",
                    "BJ" : "m 343.23,195.39 0.4,2.1 -2,3 -0.1,5.1 -1.6,0.2 -0.5,0.1 -0.1,-5.4 -1.3,-3.7 1,-0.9 1.9,-0.9 0.8,-0.9 z",
                    "NG" : "m 341.53,205.59 0.1,-5.1 2,-3 -0.4,-2.1 1,-3.4 2.6,-0.8 2.8,1.7 1.7,-0.7 3.4,1 1,-0.8 2.5,-0.2 2,0.6 2.1,-1.3 0.9,1.3 1.2,1.7 -4.7,7.4 -1.8,3.6 -1.4,-1.3 -3.3,2.5 -0.5,1.9 -4.6,1 -1.1,-0.7 -2.3,-3.2 z",
                    "IR" : "m 422.03,143.59 -1,-1.8 -0.5,-3.1 1.6,-0.6 1.2,1.4 1.3,0.3 0.5,-0.1 0.2,0.1 2.8,-1.9 0.1,2 1.7,0.9 0.2,1.5 2.3,0.8 1.7,1.4 1.6,0.4 4,-0.6 -0.2,-1.1 2.9,-1.6 3.4,-0.4 0.3,0.6 2.8,0.6 2.9,2.4 1.6,0.1 0.2,2.1 -1.4,5.5 0.5,3.3 1.7,0.2 0.2,1.1 -1.8,2.1 2,2.7 1.6,0.7 0.7,3.3 -2.4,0.8 -0.5,2.1 -3.2,-0.3 -5,-0.9 -0.5,-2.2 -1.3,-0.7 -3,1.5 -6.5,-3 -2.6,-4.7 -2.1,-0.4 -0.8,0.9 -1.6,-2.2 0.3,-1.6 -0.9,-1.3 -2.1,-1.2 -1.7,-2.2 1.8,-3.9 -2,-0.4 z",
                    "SV" : "m 168.13,192.09 -1.8,0.3 -2.5,-0.9 -0.1,-0.2 1.5,-1.1 3.2,1.1 z",
                    "SL" : "m 314.33,204.59 -2.8,-1.3 -0.7,-2.8 1.5,-1.5 2.4,-0.3 1.9,2.9 -0.6,1.4 z",
                    "GW" : "m 307.53,196.79 -1.7,-0.5 -1.6,-2.1 3,-0.7 2.8,0 0,1.9 -1.9,0.4 z",
                    "HR" : "m 371.73,131.79 -1.3,-0.9 -0.3,-0.2 0.4,0.2 1.1,0.6 0,0.1 z m -9.4,-7.2 3.3,0.1 0.6,-1.8 1.7,-0.7 1.6,1.3 3,0.1 0.2,1.8 0,0.7 -4,-1 -2.2,0.3 0.8,2.2 2.6,3 -1.7,-0.4 -3.5,-2.5 0,-2.2 -1.7,0.7 z",
                    "BZ" : "m 165.63,183.49 0.5,-0.1 1.1,-1.2 0.2,3 -1.4,2.1 -0.6,0 z",
                    "ZA" : "m 397.83,270.09 0.4,0 1.1,0 -1,3.5 -2.3,2.1 -1.5,2.7 -2.9,3 -3.4,2.7 -4.3,1.4 -4.4,-0.4 -1.3,0.7 -4.3,0.9 -2.2,-2 -1.3,-2.1 0.9,-0.4 -0.2,-2 -1.7,-2.9 -1.6,-3.6 1.2,-1.2 0.6,1.4 3.3,0.5 1.6,-1.1 0,-7.5 1.7,2.8 -0.3,1.6 1.9,-0.1 2,-1.8 0.6,-1.4 3.4,1.1 1.5,-0.4 0.5,-1.8 1.9,-0.9 0.3,-1.3 2.3,-2 2.2,-0.9 1.9,0.3 1.8,0.1 1.3,4.1 -0.1,3.1 -1,-0.5 -1.1,1.4 0.6,1.6 z m -7.6,3.8 -2.2,1.9 2,2.2 2.6,-2.8 -1.5,-1.6 z",
                    "CF" : "m 380.03,196.89 1.5,2.1 -0.3,2 1.4,0.2 2.2,2.6 2,1.3 0.1,1.1 1.9,1.9 -3.5,-0.6 -5.9,2.3 -3.7,-0.4 -1.7,-1.4 -2.2,1.5 0.1,1.7 -2.2,-0.5 -1.7,0.5 -0.7,2.3 -0.2,-1.1 -1.8,-1.8 -1.5,-4.3 2.1,-2.9 2.5,0 3.5,-1 0.9,-1.8 2.6,-0.3 3.9,-3.5 z",
                    "SD" : "m 382.63,201.19 -1.4,-0.2 0.3,-2 -1.5,-2.1 -1.2,-3.5 -0.3,-2 0.9,-0.7 1.1,-3 1.7,-0.1 0,-7.4 0,-1 1.9,0 0,-4 12,0 10.8,0 0.8,2.4 0.4,4.2 2.1,1.4 -3.1,1.8 -1.1,3.8 0.2,1.7 -0.8,3 -0.8,0.1 -1.6,3.7 -0.5,-0.3 -1,2.7 -1.7,-2.6 0.1,-2.7 -2.2,0.4 0.6,1.9 -2.2,2.4 -2.4,-0.9 -2.2,1.8 -4.4,-0.4 -1.4,-1.7 -1.4,0.2 z",
                    "CD" : "m 392.23,224.09 -0.2,3.3 0.8,3.1 2,2.9 -3.2,0.5 -1.1,1.2 0.4,0.6 0.3,0.8 -0.6,3.1 1.3,1.8 1.4,-0.4 0,2.5 -1.1,0 -3.9,-3.6 -0.9,0.8 -2.6,-0.7 -0.1,-0.9 -1.9,0.4 -0.6,-1 -3.3,0.8 -0.2,-2.6 -0.7,-1 0,-4.1 -4.3,-0.5 -0.4,1.9 -3.3,0.2 -1.2,-1.5 -0.7,-2.7 -6.6,-0.1 -1.4,0.4 -0.4,-0.6 0.4,-1.3 1.2,-0.8 2.5,-0.7 0.9,1 2.6,-2.8 0.1,-2.3 2.9,-3 0.6,-5.1 1,-2.5 -0.1,-1.7 2.2,-1.5 1.7,1.4 3.7,0.4 5.9,-2.3 3.5,0.6 1.8,1.5 2.7,-0.5 2,2 -0.2,2 1,0.5 -1.4,1.3 -0.1,0.5 -1,0.8 -0.5,1.7 -0.7,0.8 0.6,0 -0.1,0.8 0,0.9 -0.4,0.3 -0.3,0.3 -0.5,0.5 -0.1,1 0,0.3 0.2,0.1 0.4,0.7 z",
                    "KW" : "m 429.03,161.99 -1.9,-1 -1.7,-0.2 1.1,-1.8 1.6,-0.1 0.3,1.4 z",
                    "DE" : "m 353.93,119.39 0,0 0,0 z m -1,-18.4 0.3,0 1.1,0.3 3.9,2.3 2,-1.5 2.2,-0.3 0.7,1.4 0.4,0.4 -0.1,0 0.1,0.1 -0.3,-0.3 0.4,0.8 -0.3,2.2 1,0.7 0.3,4.4 -4.4,1.2 -0.2,1.6 2.7,2.4 -2,1.6 0.5,1.5 -1.6,-0.5 -2.3,0.7 -2.4,-0.3 -1.3,-0.7 0.3,0.4 -1.4,0.2 -1.7,0 1.2,-3.4 -3.6,-1.2 0.3,-0.9 -0.7,-0.8 0.5,-0.4 -0.7,-1.2 0.7,-2.7 1.2,-1 0.4,-2.6 0.2,-1.3 2.4,0.5 0.8,-0.9 z",
                    "BE" : "m 347.83,111.69 0.7,1.2 -0.5,0.4 -0.8,0.6 0.2,0.9 -3.2,-1.1 -3.1,-2.8 1.2,-0.6 0.4,-0.1 0.9,0.4 0.8,-0.4 0.1,0.1 -0.1,-0.1 1.5,-0.3 z",
                    "IE" : "m 322.43,100.59 -1.8,1.6 1.7,0.9 2,0 0.5,3 -0.7,2 -2.3,0.3 -2.2,1.3 -2.1,0.3 -1.2,-1.2 2.9,-3.5 -2.1,-0.3 0.3,-2.5 2.4,0.2 0.6,-2.4 z",
                    "KP" : "m 586.53,132.09 -2,1.7 0.1,1.6 -4.2,2.5 -0.1,1.3 1.7,1.2 -2.3,0.7 -0.9,1 -2.1,-0.4 -0.4,0.8 -1.1,-1.9 1.3,-1.8 -2.1,-1.7 3.1,-1.8 1.2,-1.8 2.9,0.7 -0.2,-1.4 3.2,-1.1 0.3,-1.2 1.3,1 0.2,0.3 z",
                    "KR" : "m 578.83,142.09 0.9,-1 2.3,-0.7 1.9,2.9 0.2,3.9 -1.6,1.7 -1,-0.4 -1.1,1.1 -2.2,-0.4 z",
                    "GY" : "m 221.43,201.49 1.7,1 3.7,3.7 -0.1,1.1 -1.6,2.5 3,4.3 -1.2,-0.2 -3.2,1.6 -1.8,-1.3 -0.5,-1.6 0.9,-2.3 -0.9,-2.2 -1.4,-0.2 -1.2,-1.4 0.3,-1.2 1.7,-1.1 -0.4,-1.3 z",
                    "HN" : "m 167.43,187.59 4.4,-0.5 3.1,0.3 2.2,1.6 -3.2,0.4 -1.8,1.6 -1.9,0.4 -1.1,1.5 -0.4,-0.5 -0.6,-0.3 0.3,-0.8 -3.2,-1.1 0.4,-1.3 z",
                    "MM" : "m 512.93,177.09 0.7,-0.5 -0.1,-1.3 1.2,-0.8 0.2,-3.3 1.5,0.3 1.8,-3.8 0,-1.5 3.2,-1.8 1.2,-1.8 0.4,-0.6 2.2,1.9 0,3.5 -1.9,1.7 -0.1,2.4 1.8,-0.4 0.6,1.8 1.2,0.2 -0.7,1.8 1.5,0.2 0.5,1.3 1.9,-0.3 -0.9,0.5 -1.2,2 -3,1.3 -0.9,-0.3 -1.2,2.6 2.7,4.6 -1.1,2.6 1.7,2.1 0.9,3.7 -1.7,2.8 -0.3,-5.5 -1.5,-3.2 -0.3,-2.8 -0.8,-1.4 -3.4,2.5 -2.4,-0.6 0.8,-2.9 -1.2,-3.7 -1.8,-1.2 z",
                    "GA" : "m 355.03,215.89 3,0 0,-2.3 0,-0.2 3.7,0.3 -0.1,1.7 2.4,0.6 -1.2,2.2 1.2,0.7 -0.7,3.6 -3.2,-1.1 -1.7,0.8 0.5,2.6 -1.3,0.5 -3.6,-4 -1.1,-2.4 1.1,-0.3 0.1,-1.9 z",
                    "GQ" : "m 358.03,213.59 0,2.3 -3,0 -0.8,-0.4 0.8,-2.2 0.4,0.3 z",
                    "NI" : "m 172.23,196.59 -3.7,-3.4 0.6,-0.3 1.1,-1.5 1.9,-0.4 1.8,-1.6 3.2,-0.4 -1.4,7 0.4,0.9 -1.3,-0.1 z",
                    "LV" : "m 388.63,93.89 0.6,1.9 1,1.9 -1.1,0.9 -1.9,0.4 -3.3,-2.1 -0.8,0.5 -4.5,-0.5 -2,1 0,-2.1 1.3,-2 1.7,-0.5 2,2.1 1.4,-0.7 -0.2,-1.7 1.9,-0.6 2.3,1.5 z",
                    "UG" : "m 397.23,219.69 0,-0.1 0,0.1 z m -4.3,0.7 0,-0.9 0.1,-0.8 0.5,-0.6 -0.4,-0.2 0.5,-1.7 1,-0.8 1.8,-1.3 -0.3,-0.5 -1,-0.5 0.2,-2 5.1,-0.5 0.9,-0.9 1.8,3.4 0.1,1.5 -1.8,2.7 -2.7,0.2 -1.6,2.2 -1.8,0 -0.7,0.1 -1,0.8 z",
                    "MW" : "m 403.13,239.59 0,-0.1 0,0 z m -0.3,-0.3 0,0 0,-0.1 0,0.1 z m -0.9,-3 -0.2,-0.2 0,0 z m -0.4,-0.4 0,0 0,0 z m -2.2,-0.2 0.9,0.4 1,0.2 0.8,3.8 0,3.2 1,0.3 1.9,2.2 -0.1,2.7 -1.3,1 -1.7,-1.3 0.2,-2.9 -1.3,0.4 -0.9,-1.1 -0.7,-0.6 1,-2.3 -0.2,-3.2 0.8,-0.5 z",
                    "AM" : "m 425.33,139.79 -0.2,-0.1 -0.5,0.1 -0.7,-1.5 -1.9,-0.3 -2.1,-1 -0.5,-2.2 1.5,-0.3 1.5,-0.1 1.9,2.4 -0.6,0.4 z m -2.8,-4.9 0,0 -0.1,0 0,0.1 z m 0.4,0.2 -0.1,-0.1 0,0 0,0.1 z m 0.5,0.8 0,0 0.1,-0.1 -0.1,0 z",
                    "SX" : "m 419.13,195.79 1.9,2 2.9,-0.8 1.3,0.4 1.8,-1 3,-0.1 0,3.4 -1.9,2.8 -1.9,0 -5.6,-1.9 -2.7,-3.1 0.6,-0.7 z",
                    "TM" : "m 445.53,132.89 1.5,0.6 -0.4,-1.1 1.8,-1.5 2.5,1.1 0.4,2.1 3.6,0.9 0.9,2.3 6.1,3.9 1.8,0.5 -0.1,1.5 -1.4,-0.5 -1.9,1.1 -0.6,1.8 -4.2,2.4 -1.9,-1 -0.2,-2.1 -1.6,-0.1 -2.9,-2.4 -2.8,-0.6 -0.3,-0.6 -3.4,0.4 -2.9,1.6 0.1,-3.5 -0.8,-2.3 -1.6,-0.2 0.4,-2.1 1.5,0.8 1.9,-1.1 -1.5,-2.2 -0.8,-0.1 0,0 1.3,-0.5 2.6,2.4 0.9,-0.1 1.9,0.1 z m -7.6,-0.1 0.2,-0.3 0.6,0 0,0 -0.8,0.3 0,0 z m 0,0 -0.5,1.3 -0.7,-0.8 1.2,-0.5 0,0 z",
                    "ZM" : "m 390.93,235.69 1.2,-1.4 -0.5,-0.4 3.2,-0.5 0.9,1.1 0.2,-0.4 1.6,1 1.8,0.6 1.4,2.3 -0.8,0.5 0.2,3.2 -1,2.3 0.7,0.6 -5.7,1.9 0.4,1.3 -3,0.8 -0.2,0.9 -1.2,0.6 -1.1,0.6 -1,1.6 -0.6,0.2 -2.8,-0.5 -2,-0.6 -1.6,0.3 -2.6,-2.9 0,-6.2 3.8,0 0,-4.1 0.6,1 1.9,-0.4 0.1,0.9 2.6,0.7 0.9,-0.8 3.9,3.6 1.1,0 0,-2.5 -1.4,0.4 -1.3,-1.8 0.6,-3.1 z",
                    "NC" : "m 401.23,148.19 -0.3,0 -0.1,0 -0.6,-0.3 -1.2,0 0,0 0,0 0,0 -0.1,0 0,0 3.6,-1.1 -1.1,1.4 z m -2.5,-0.3 0.1,0 -0.1,0.1 z",
                    "MR" : "m 319.73,164.49 6,3.8 1.3,0.9 -3.3,0 2,18.9 -10,0.1 -1.9,-0.2 -1,1.5 -4,-3.7 -3.8,0.4 -0.4,1.2 1,-4.2 -1,-2.8 0.7,-1.6 -1.7,-1.1 0.2,-1.1 7.6,0 0,-3.4 1.9,-0.9 0,-5.1 6.3,0 z",
                    "DZ" : "m 327.03,169.19 -1.3,-0.9 -6,-3.8 0,-0.7 0,-2.1 1.9,-1.5 3.7,-0.5 4,-2.3 0,-1.9 2.2,-1.1 2.4,0.1 0.4,-0.9 -1.2,-1.5 -0.2,-3.2 -0.9,-0.8 6.3,-3 3.6,-0.7 3.5,-0.2 0.9,0.5 4.1,-0.9 2.3,0.3 -0.4,1.1 -0.3,3.9 -1.4,1.6 1.6,2.9 1.4,1 0.9,3.8 0.6,2.3 0,5.4 -0.8,0.8 1.2,2.6 2.9,1.1 0.8,1.6 -8.6,5.3 -3.2,2.8 -3,0.6 -2.2,0 0.2,-1.3 -2.5,-0.8 -1.4,-1.8 z",
                    "LT" : "m 377.03,100.09 -0.2,-1.2 -0.2,-1 2,-1 4.5,0.5 0.8,-0.5 3.3,2.1 0.3,0.9 -1.7,1 -0.8,1.7 -2,1.1 -1.8,-0.1 0,-0.6 -1.3,-0.5 0.1,-1.4 z m -0.7,-0.1 0.3,-0.6 -0.1,0.6 z",
                    "ET" : "m 416.33,196.19 -0.2,0.2 0.2,0.2 0.3,0.3 1.9,-0.1 -0.6,0.7 2.7,3.1 5.6,1.9 1.9,0 -5.8,5.9 -1.8,-0.1 -4,1.9 -2.2,-0.6 -2.3,1.7 -2.8,-0.4 -2.4,-1.6 -1.7,-0.3 -2.3,-3.9 -2,-1.9 -1.2,-0.3 0.3,-1.3 1.4,0.1 0.2,-2 1,-2.7 0.5,0.3 1.6,-3.7 0.8,-0.1 0.8,-3 2.6,-1.2 5.7,1.5 2.9,3.1 -1.2,1.8 z",
                    "ER" : "m 417.43,193.89 -2.9,-3.1 -5.7,-1.5 -2.6,1.2 -0.2,-1.7 1.1,-3.8 3.1,-1.8 2.1,5.6 2.8,0.9 3.7,3.8 -0.8,0.6 z",
                    "GH" : "m 338.53,206.19 -6.2,2.6 -2,-0.7 0,-0.1 0.3,0.1 0.2,-0.1 -0.7,-3.3 1.4,-2.6 -0.4,-2.4 -0.1,-2.9 5,-0.3 1,1.8 0.3,6.4 z",
                    "SI" : "m 362.33,124.59 0,-0.1 0.2,-0.1 -0.6,-1.7 0.6,-0.6 1.6,0.4 3,-1.2 0.3,0 0.5,0.9 -1.7,0.7 -0.6,1.8 z",
                    "GT" : "m 166.03,187.29 0.6,0.3 0.8,0 -1.8,1.3 -0.4,1.3 -1.5,1.1 0.1,0.2 -2.4,-0.4 -1.7,-1.2 0.9,-2.9 2.6,-0.6 -2,-1.8 0.9,-1.1 3.5,0 -0.2,3.8 z",
                    "BA" : "m 370.13,130.69 -0.2,-0.1 0,0 -2.6,-3 -0.8,-2.2 2.2,-0.3 4,1 0.7,-0.1 -0.4,3.2 -1.4,1.3 0,1 -1.1,-0.6 z",
                    "JO" : "m 404.13,156.59 0.2,-0.2 -0.2,-0.1 -0.1,0 0.1,-0.2 0.2,0 0.1,-0.9 0,-1.8 0.3,-0.3 2,1 3.8,-2.3 0.6,1.9 0.1,0.8 -4.2,1.3 2,2.1 -3.7,2.7 -2.1,-0.3 0.1,-0.4 -0.1,-0.1 0.4,-1.8 z",
                    "SY" : "m 410.53,151.79 -3.8,2.3 -2,-1 0.3,-0.4 -0.1,-1 1.5,-1.7 -1.3,-0.9 -0.1,-1.7 0,-1.1 1.4,-2 3.1,-0.1 2.9,0.3 4.7,-1.3 0.3,0.5 -1.9,1.3 -0.3,3.8 -1,1 z",
                    "MC" : "m 350.53,128.69 0,0.1 -0.1,0 0.1,-0.1 z",
                    "AL" : "m 376.43,135.29 0,0 0,-0.1 z m -3,-2.8 0,-0.3 0,-0.2 0.6,-0.8 0.7,0.3 0.8,0.8 0.2,0.7 -0.3,0.8 0.4,1 0,0.4 0.2,0 0.2,0 0.2,0 -0.1,0.1 0.1,0 0,0.1 -0.1,0.1 0.1,0 0,0.1 0.1,0.1 0,0.1 0,-0.1 -1.2,2 -0.7,0.3 -1.4,-2.2 0.2,-2.7 -0.1,-0.3 z m 0,-0.4 0,0 -0.1,0 z",
                    "UY" : "m 234.53,282.19 -0.9,1 0.2,0.9 0,0.2 0.3,0.2 -0.8,1.4 -2.2,1.2 -2.3,0 -3.3,-1 -1,-1.2 0.3,-3.1 0,-1.2 1.2,-3.6 1.5,-0.2 1.5,2.1 0.8,-0.5 z",
                    "CNM" : "m 401.43,148.19 0,0 0,0 -0.2,0 0,0 z m -2.7,-0.3 0,0 0,0.1 0.1,-0.1 0,0 0,0.1 z m 0.2,0 0,0 0,0 0.1,0 0,0 z m 1.9,0.3 0,0.1 0,0 -0.5,-0.3 -1.3,-0.1 0,0 1.2,0 z",
                    "MN" : "m 561.33,119.09 0.7,-0.2 -0.3,-0.3 1.5,0 2.6,2.6 -0.4,0.8 -2.7,-0.3 -3.2,1 -1.8,2 -2.1,0.2 -1.8,1.5 -3.1,-0.8 -1.1,1.5 1,1.7 -4.7,2.9 -5.2,0.4 -3,1.2 -2.8,0 -3.3,-1 -0.7,-0.7 -10.1,-0.5 -1.9,-3.7 -3.6,-1.5 -5,-0.6 0.3,-3.3 -1.4,-2.6 -2.4,-0.8 -2.1,-1.5 -0.3,-1.4 8.7,-4.2 3.6,0.7 0.7,1.4 5.2,0.7 1.9,-2 -1,-1.2 2.1,-2.9 6.3,2.1 0.2,1.8 2.7,1.2 3.1,-0.9 5,1.4 1.2,1.5 4.2,0.5 4.4,-1.2 2.4,-1.7 2.1,1 2.5,0.2 -2.3,4.1 0.7,1.1 z",
                    "RW" : "m 391.83,222.89 -0.2,-0.1 0,-0.3 0.9,-0.9 -0.3,-0.6 0.3,-0.3 0.4,-0.3 0.7,0.2 1,-0.8 0.7,1 -0.5,1.5 -1.3,0.5 z",
                    "SO" : "m 416.53,210.19 4,-1.9 1.8,0.1 5.8,-5.9 1.9,-2.8 0,-3.4 4.5,-1.1 -0.9,4.5 -5.5,9.6 -3.7,3.8 -4.9,3.5 -3.7,4.4 -1.1,-1.6 0,-7 z",
                    "BO" : "m 204.03,248.89 -0.1,-0.3 0.3,0.3 -0.1,0 z m -0.8,2.5 -0.3,-0.4 1.2,-1.5 0.9,-0.4 -1.5,-1.6 0.9,-2.6 -0.4,-1 0.8,-2.2 -1.7,-3 2.4,0 3.2,-2 2.3,-0.5 0.1,2.9 1.9,2.5 2.5,0.4 2.4,1.7 2.6,0.5 0.6,4.8 3.2,0.1 0.2,1.8 1.6,1.8 -1.2,3.9 -1.8,-1.7 -5.1,0.7 -0.9,1.8 -0.8,3.4 -2.4,-0.5 -0.8,1.7 -0.5,-1.3 -3.1,-0.8 -1.9,2 -1.3,0 -0.6,-3 -1.1,-1.7 0.6,-2 -1,-1 z",
                    "CM" : "m 361.73,213.69 -3.7,-0.3 0,0.2 -2.6,0 -0.4,-0.3 -0.3,-2.9 -2,-1.8 0.5,-1.9 3.3,-2.5 1.4,1.3 1.8,-3.6 4.7,-7.4 -1.2,-1.7 0.5,-0.1 -0.1,0.4 0.4,-0.1 1.3,2.7 -0.2,1.5 1.2,1.5 -2.9,0 -0.4,0.7 2.3,2.2 0.6,1.8 -2.1,2.9 1.5,4.3 1.8,1.8 0.2,1.1 -0.1,1 -3,-0.9 z m 2.1,-21 0,0.1 0,-0.1 z",
                    "CG" : "m 361.73,213.69 2.5,-0.1 3,0.9 0.1,-1 0.7,-2.3 1.7,-0.5 2.2,0.5 -1,2.5 -0.6,5.1 -2.9,3 -0.1,2.3 -2.6,2.8 -0.9,-1 -2.5,0.7 -0.6,-0.5 -1.4,1.2 -0.4,-0.8 -1.3,-1.2 1.3,-0.5 -0.5,-2.6 1.7,-0.8 3.2,1.1 0.7,-3.6 -1.2,-0.7 1.2,-2.2 -2.4,-0.6 z",
                    "EH" : "M 319.73047,163.78906 319.67383,164 H 311.125 l -0.89453,1.78906 -1.70117,1.10156 -0.69922,2.89844 -1.40039,1.40039 -2.69922,5.20117 -0.20117,1.09961 0.10156,0.19922 0.19922,-1.09961 h 7.59961 v -3.40039 l 1.90039,-0.90039 v -5.09961 h 6.30078 l 0.0996,-2.69922 z",
                    "RS" : "m 373.03,129.19 0.4,-3.2 -0.7,0.1 0,-0.7 -0.2,-1.8 1.2,-0.6 1.3,0.1 2.4,2.3 -0.2,0.9 2.5,1.3 -0.6,1 1.1,1.6 -1.1,1.8 -0.8,0 -0.7,0.2 0.3,-1 -1.9,-1.4 -0.8,1 -1.4,-0.8 z",
                    "ME" : "m 373.03,129.19 0.8,0.8 1.4,0.8 -0.6,0.3 0.1,0.4 -0.7,-0.3 -0.6,0.8 0,0.1 -0.1,0 -0.4,0 0.5,0.4 -0.1,0.3 0.1,0.3 -1.2,-1 -0.5,-0.3 -0.1,-0.2 0,-0.1 0,-1 z",
                    "TG" : "m 338.03,196.79 1.3,3.7 0.1,5.4 -0.4,0.1 -0.5,0.2 -1.2,-1.5 -0.3,-6.4 -1,-1.8 1.2,0.3 z",
                    "LA" : "m 542.13,189.59 -1.3,0.8 -3.1,-0.1 0.8,-2.7 -1.7,-1.6 0.2,-1.6 -1.6,-1.9 -1.2,-0.2 -1.5,1.2 -1,-0.8 -1.8,1.5 0.2,-4 -1.5,0 -0.7,-1.6 1.2,-2 0.9,-0.5 1,0.9 -0.4,-2.3 1.2,-0.3 1.6,1.7 0.3,1.4 1.8,-0.2 1.7,1.8 -2.1,1.5 1.7,1 3.4,3.5 2.2,3.4 z",
                    "AF" : "m 479.73,143.39 -1,0.3 0.3,0.2 -3.8,0.4 -2.7,1.7 0.9,1.9 -1.1,2.4 -2,0.1 0.6,1.5 -1.5,0.5 -0.4,2.5 -2.3,0.2 -3.4,1.9 -0.3,2.3 -4,0.9 -3.1,0 -3.1,-1 1.8,-2.1 -0.2,-1.1 -1.7,-0.2 -0.5,-3.3 1.4,-5.5 1.9,1 4.2,-2.4 0.6,-1.8 1.9,-1.1 1.4,0.5 1.4,0.3 1,0 2.9,0.2 1.6,-0.9 0.9,-1.8 2,1 0,2.6 3.2,-1.7 z",
                    "UA" : "m 378.73,117.59 0.3,-0.7 0.4,-1 0.2,-1.1 2.8,-2.5 -0.9,-2.5 1.5,-0.9 2.6,-0.1 2.9,0.8 3.9,0.6 2.3,-0.5 0.8,-1.3 1.6,-0.1 3.9,-0.6 1.2,1.5 -0.5,1.3 1.7,0.1 1.1,2.2 3.4,-0.2 3.4,1.4 1.8,1.5 -0.8,1.2 0.1,2.3 -1.8,-0.1 -1.2,1.8 -4.4,1.1 -2.1,1.2 1.5,2.5 -3.1,1.7 -2.1,-2.9 1.7,-0.7 -3.9,-0.7 1.2,-0.6 -2.3,-0.4 -2.6,2.4 -0.2,1 -1.7,-0.1 -1.1,-0.5 1.4,-2.4 2.2,0.2 -1.9,-2.8 0.3,-0.8 -2.9,-1.4 -2.2,0.4 -2.6,0.9 -4.5,-0.1 -0.5,-0.4 z m 15.3,4.9 0.5,0.7 0.2,-0.2 -0.6,-0.6 z m 7.4,0.7 1.8,1 -0.3,-0.9 -1.2,-0.6 z",
                    "SK" : "m 379.43,115.89 -0.4,1 -0.3,0.7 -3.2,-0.3 -1.1,0.9 -4,1 -1.3,-0.6 -0.6,-0.9 0.2,-0.6 2.2,-1.2 1.4,-1.1 5.8,0.4 z",
                    "JK" : "m 485.23,147.19 -1.4,0.9 -0.5,-1.2 1.1,0.4 z",
                    "BG" : "m 391.03,128.69 -1.7,3.1 0.6,1 -1.8,-0.2 -1.4,0.8 -2.1,1.1 -1.4,-0.8 -3,0.6 0.1,-1 -1.2,-1.3 1.1,-1.8 -1.1,-1.6 0.6,-1 1.2,0.8 3.9,0.5 3.2,-1.2 z",
                    "QA" : "m 433.63,169.69 -0.2,-1.5 1.6,-0.8 -0.7,2.6 -0.4,0.1 z",
                    "LI" : "m 354.53,120.39 0.2,0.3 -0.1,0.2 -0.2,-0.1 z",
                    "AT" : "m 354.63,120.89 0.1,-0.2 -0.2,-0.3 0.3,-0.4 -0.2,-0.2 0.2,0 0.1,-0.1 2.4,0.3 2.3,-0.7 1.6,0.5 -0.5,-1.5 2,-1.6 1.7,0.4 0.5,-0.9 3.8,0.9 -0.2,0.6 0.6,0.9 -1.1,0.6 -0.9,2.1 -3,1.2 -1.6,-0.4 -2.5,-0.4 -0.4,-0.9 -3.3,0.5 -0.7,0.1 z",
                    "SZ" : "m 397.43,268.29 0.2,0.6 0.2,1.2 -1.9,0.7 -0.6,-1.6 1.1,-1.4 z",
                    "HU" : "m 378.73,117.59 0.9,0.7 0.5,0.4 -1.7,1.1 -1.6,2.9 -1.8,0.4 -1.3,-0.1 -1.2,0.6 -3,-0.1 -1.6,-1.3 -0.5,-0.9 -0.3,0 0.9,-2.1 1.1,-0.6 1.3,0.6 4,-1 1.1,-0.9 z",
                    "RO" : "m 379.73,127.59 -2.5,-1.3 0.2,-0.9 -2.4,-2.3 1.8,-0.4 1.6,-2.9 1.7,-1.1 4.5,0.1 2.6,-0.9 0.8,0.3 2.4,3.7 -0.1,2.8 1.1,0.5 1.7,0.1 -2,2.1 -0.1,1.3 -3,-1 -3.2,1.2 -3.9,-0.5 z",
                    "NE" : "m 343.23,195.39 -1.5,-1.3 -0.8,0.9 -0.4,-1.3 -2.3,-0.9 -1.5,-3.6 2.1,-0.7 4.2,-0.2 1.3,-2 0.1,-5.4 3,-0.6 3.2,-2.8 8.6,-5.3 2.9,0.7 1.4,1.1 1.5,-0.8 0.3,3.1 1.6,2.2 -0.5,0.9 -0.5,5.9 -3.9,4.9 0.3,1.3 -2.1,1.3 -2,-0.6 -2.5,0.2 -1,0.8 -3.4,-1 -1.7,0.7 -2.8,-1.7 -2.6,0.8 z",
                    "LU" : "m 348.43,114.99 -0.7,0 -0.3,-0.2 -0.2,-0.9 0.8,-0.6 0.7,0.8 z",
                    "AD" : "m 339.03,131.39 0.6,-0.1 -0.1,0.3 -0.3,0.2 z",
                    "CI" : "m 331.13,199.69 0.4,2.4 -1.4,2.6 0.7,3.3 -5.7,0.2 -3.3,1.3 0.1,-2.8 -2.1,-1.3 0.2,-2 1.2,-3.6 -0.2,-1.4 3.3,-0.2 1.4,-0.3 1.6,1.4 2.8,-0.5 z m -0.8,8.4 -0.3,-0.1 0.3,0 z",
                    "LR" : "m 320.03,203.39 -0.2,2 2.1,1.3 -0.1,2.8 -3.3,-1.5 -4.2,-3.4 1.7,-1.6 0.6,-1.4 1.5,0.3 0.7,2.2 z",
                    "BN" : "m 556.53,208.59 0.2,-0.1 0,-0.1 0.4,1.2 z m -2,0.4 1.1,-0.2 0.8,-0.3 -0.7,1.6 z",
                    "IQ" : "m 429.23,158.99 -0.8,-0.2 -0.3,0.1 -1.6,0.1 -1.1,1.8 -3.6,-0.2 -5,-3.9 -3.1,-1.8 -2.5,-0.4 -0.1,-0.8 -0.6,-1.9 3.7,-2 1,-1 0.3,-3.8 1.9,-1.3 0.7,-0.5 3.9,0.4 1,2.6 2,0.4 -1.8,3.9 1.7,2.2 2.1,1.2 0.9,1.3 -0.3,1.6 z",
                    "GE" : "m 422.43,134.39 -1.5,0.1 -1.5,0.3 -1.2,-1.1 -2.4,0.2 -0.1,-2.6 -2.9,-1.8 5.5,0.6 2.1,1.4 1.7,-0.4 3.1,1.9 0.1,1.9 z",
                    "GM" : "m 304.23,192.79 -0.1,-0.6 0.5,-0.4 3.2,-0.4 -1.8,1.2 z",
                    "CH" : "m 353.93,119.39 0,0 0,0 0,0 0.6,0.4 0.1,0 0.2,0.2 -0.3,0.4 -0.1,0.4 0.2,0.1 1,0.5 0.7,-0.1 -0.9,1.2 -2.2,0.7 -1.5,-0.4 -2,0.8 -0.5,-0.5 0.1,-0.7 -0.3,-0.3 -0.5,0.4 -0.3,0.2 -0.2,-0.7 2.8,-2.4 1.7,0 z",
                    "TD" : "m 364.03,192.99 0.6,-0.2 -0.8,-0.1 0,0 -0.2,-0.1 0.1,0.1 -0.5,0.1 -0.9,-1.3 -0.3,-1.3 3.9,-4.9 0.5,-5.9 0.5,-0.9 -1.6,-2.2 -0.3,-3.1 1.9,-0.9 15.3,7.9 0,7.4 -1.7,0.1 -1.1,3 -0.9,0.7 0.3,2 1.2,3.5 -0.7,-0.1 -3.9,3.5 -2.6,0.3 -0.9,1.8 -3.5,1 -2.5,0 -0.6,-1.8 -2.3,-2.2 0.4,-0.7 2.9,0 -1.2,-1.5 0.2,-1.5 z",
                    "KV" : "m 375.73,132.99 -0.2,-0.7 -0.8,-0.8 -0.1,-0.4 0.6,-0.3 0.8,-1 1.9,1.4 -0.3,1 -0.9,0.1 z",
                    "LB" : "m 403.53,152.39 0.9,-1.7 0.7,-1.6 1.3,0.9 -1.5,1.7 -0.7,0.7 z",
                    "DJ" : "m 418.53,196.79 -1.9,0.1 -0.3,-0.3 0.2,-0.2 -0.2,-0.2 -0.1,-0.5 1.2,-1.8 0.6,0.2 0.8,-0.6 0.5,1.4 -0.2,0.9 z",
                    "BI" : "m 393.13,226.19 -0.6,-1 -0.3,-1.1 0,-0.5 -0.4,-0.7 1.7,-0.1 1.3,-0.5 0.5,1.6 z",
                    "SR" : "m 231.73,213.29 -2.6,-0.3 -1,1.1 -3,-4.3 1.6,-2.5 0.5,-1 5.6,0.4 -0.2,0.9 -0.6,0.8 0.9,2.8 z",
                    "IL" : "m 401.83,156.19 1.2,-2.1 0.5,-1.7 0.7,0 0.7,-0.7 0.1,1 -0.3,0.4 -0.3,0.3 0,1.8 -0.4,0.5 0.1,0.4 -0.1,0.2 0.1,0 -0.1,0.2 0.1,0.1 -0.5,1.4 -0.4,1.8 -0.1,0.2 0,0 -0.1,-0.5 z",
                    "ML" : "m 325.73,197.89 -1.4,0.3 -3.3,0.2 -0.8,-2.3 -1.5,-2.2 -2.9,1.1 -1.3,-0.9 -1.4,-2.6 -0.3,-2 1,-1.5 1.9,0.2 10,-0.1 -2,-18.9 3.3,0 11.5,7.8 1.4,1.8 2.5,0.8 -0.2,1.3 2.2,0 -0.1,5.4 -1.3,2 -4.2,0.2 -2.1,0.7 -1.8,-0.3 -2.7,1.8 -1.4,0.2 -1.3,1.7 -1,-0.5 -1,2.4 -1.1,0.4 z",
                    "SN" : "m 304.23,194.19 -0.1,-0.9 0.1,-0.5 1.8,-0.2 1.8,-1.2 -3.2,0.4 -1.1,-2.6 1.1,-1.8 0.4,-1.2 3.8,-0.4 4,3.7 0.3,2 1.4,2.6 -1.9,0.1 -2.6,-0.7 -2.8,0 z",
                    "GN" : "m 314.53,194.09 1.3,0.9 2.9,-1.1 1.5,2.2 0.8,2.3 0.2,1.4 -1.2,3.6 -1.2,0.7 -0.7,-2.2 -1.5,-0.3 -1.9,-2.9 -2.4,0.3 -1.5,1.5 -1,-1.5 -2.3,-2.2 0.6,-1 1.9,-0.4 0,-1.9 2.6,0.7 z",
                    "ZW" : "m 384.63,251.99 2.8,0.5 0.6,-0.2 2.1,-2.2 0,0 1.2,-0.6 0.2,-0.9 3,-0.8 4.9,2 0.1,3.3 -0.5,2.2 0.5,1.1 -1.2,2.5 -2.1,2.2 -1.8,-0.1 -1.9,-0.3 -2.6,-1.2 -0.6,-2.2 -3,-2 z",
                    "PL" : "m 373.83,102.19 -0.4,0.5 0.7,-0.5 5.6,0.3 0.2,0 1.3,0.5 0,0.6 0.8,3.3 -1.4,0.9 0.9,2 0.9,2.5 -2.8,2.5 -0.2,1.1 -1.3,-0.7 -5.8,-0.4 -0.5,-0.9 -4.1,-1.7 -3.1,-0.7 -0.3,-4.4 -1,-0.7 0.3,-2.2 0.7,-0.4 -0.8,-0.1 -0.1,-0.1 0.1,0 3.7,-1 0.8,-0.7 3.3,-0.7 0.6,1.1 1.9,-0.1 z",
                    "MK" : "m 376.43,135.29 0,0 0,0.1 z m 2.7,-3.3 1.2,1.3 -0.1,1 -2,0.5 -1.5,0.6 -0.3,-0.4 0,0.3 0,-0.1 0,0 -0.2,0 -0.2,0 0.1,-0.4 -0.3,0 -0.4,-1 0.3,-0.8 1,-0.7 0.9,-0.1 0.7,-0.2 z",
                    "PY" : "m 232.43,264.39 -0.3,0.8 -0.4,2 0.1,0.2 -0.1,0.1 -0.3,2.2 -1.9,1.6 -5.4,-0.2 2,-3.9 -4.8,-2.9 -1.8,-0.4 -3.2,-3.1 0.8,-3.4 0.9,-1.8 5.1,-0.7 1.8,1.7 0.7,1.6 -0.3,2.2 3,0.1 1.5,1.1 0.4,2.6 z",
                    "BY" : "m 381.53,109.79 -0.9,-2 1.4,-0.9 -0.8,-3.3 1.8,0.1 2,-1.1 0.8,-1.7 1.7,-1 -0.3,-0.9 1.9,-0.4 1.1,-0.9 5.2,1.4 1.7,4.8 1.8,1 -2.8,1.1 1,2.3 -1.6,0.1 -0.8,1.3 -2.3,0.5 -3.9,-0.6 -2.9,-0.8 -2.6,0.1 z",
                    "CZ" : "m 372.33,114.79 -1.4,1.1 -2.2,1.2 -3.8,-0.9 -0.5,0.9 -1.7,-0.4 -2.7,-2.4 0.2,-1.6 4.4,-1.2 3.1,0.7 4.1,1.7 z",
                    "BF" : "m 338.03,196.79 -0.8,0 -1.2,-0.3 -5,0.3 0.1,2.9 -1,-0.9 -2.8,0.5 -1.6,-1.4 0.7,-3 1.1,-0.4 1,-2.4 1,0.5 1.3,-1.7 1.4,-0.2 2.7,-1.8 1.8,0.3 1.5,3.6 2.3,0.9 0.4,1.3 -1.9,0.9 z",
                    "NA" : "m 381.03,251.69 1.6,-0.3 2,0.6 -3.2,1.3 -0.5,-0.9 -4.5,0.6 0.1,7.2 -2,0.1 0,5.5 0,7.5 -1.6,1.1 -3.3,-0.5 -0.6,-1.4 -1.2,1.2 -2.2,-2.6 -1.4,-5.6 -0.4,-4.6 -1.9,-2.9 -1.6,-3.5 -1,-1.1 -0.5,-2.5 2.7,-0.6 1.5,0.9 8.6,0 0.6,0.7 3.9,0.5 z",
                    "LY" : "m 365.03,173.19 -1.5,0.8 -1.4,-1.1 -2.9,-0.7 -0.8,-1.6 -2.9,-1.1 -1.2,-2.6 0.8,-0.8 0,-5.4 -0.6,-2.3 1.4,-1.4 -0.3,-1.1 2.8,-2.2 -0.1,-1.5 1.6,0.8 1.9,-0.2 3.5,1.1 1.7,2.3 2.5,0.4 3.3,1.8 1.3,-0.5 0.3,-2.7 1.2,-1.7 2.1,-0.8 2.8,0.7 0,0.6 3.6,0.8 0.3,0.6 -0.9,3.1 0.6,2.1 0,14.6 0,4 -1.9,0 0,1 -15.3,-7.9 z",
                    "TN" : "m 358.33,152.19 0.1,1.5 -2.8,2.2 0.3,1.1 -1.4,1.4 -0.9,-3.8 -1.4,-1 -1.6,-2.9 1.4,-1.6 0.3,-3.9 0.4,-1.1 2.2,-0.9 2.7,4.6 -1.9,2 0.4,1.3 1.4,-0.2 z",
                    "BT" : "m 506.53,164.49 2.5,-2.2 2.7,1.3 0.9,1.7 -4.4,0.4 z",
                    "MD" : "m 390.33,124.69 0.1,-2.8 -2.4,-3.7 -0.8,-0.3 2.2,-0.4 2.9,1.4 -0.3,0.8 1.9,2.8 -2.2,-0.2 z",
                    "SS" : "m 401.33,209.69 -0.9,0.9 -5.1,0.5 -2,-2 -2.7,0.5 -1.8,-1.5 -1.9,-1.9 -0.1,-1.1 -2,-1.3 -2.2,-2.6 1.7,-3.1 1.4,-0.2 1.4,1.7 4.4,0.4 2.2,-1.8 2.4,0.9 2.2,-2.4 -0.6,-1.9 2.2,-0.4 -0.1,2.7 1.7,2.6 -0.2,2 -1.4,-0.1 -0.3,1.3 1.2,0.3 2,1.9 2.3,3.9 -1,-0.8 z",
                    "BW" : "m 384.63,251.99 1.7,3.3 3,2 0.6,2.2 2.6,1.2 -2.2,0.9 -2.3,2 -0.3,1.3 -1.9,0.9 -0.5,1.8 -1.5,0.4 -3.4,-1.1 -0.6,1.4 -2,1.8 -1.9,0.1 0.3,-1.6 -1.7,-2.8 0,-5.5 2,-0.1 -0.1,-7.2 4.5,-0.6 0.5,0.9 z",
                    "BS" : "m 191.33,169.99 -1.3,-1.4 0,0 0,0 0.5,0.3 0.8,1.1 z m 0,0 0.8,0.9 0.5,2.1 4.4,1.5 -4,-1 -2.2,-1.6 1.1,-1.1 -0.6,-0.8 z m -1.3,-1.4 -1.6,-1.2 -1.4,1.5 0.7,2.9 -1.3,-1.9 0.1,-1.1 1.9,-1.9 1.6,1.7 0,0 z",
                    "NZ" : "m 661.13,313.49 -2.7,-0.1 -0.2,1.4 -0.8,-2.2 -1.8,0.1 -0.6,-1 0.9,-1.6 2.7,-2.6 2.6,-1.1 2.7,-2.2 1.9,-3.9 1.2,-0.9 0.7,1.8 1.4,-0.9 0.7,2.2 -3.4,4.3 -2.1,1.3 -1.3,3.3 z m 12.3,-12.7 -1.6,1.1 -1.2,-0.7 1.2,-2.3 -2.8,-2 1.6,-1.3 0.4,-2.3 -0.9,-2.5 -2.4,-3.1 1.5,-0.5 2.4,2.7 1.4,1.1 0.2,2 2.8,0.7 2.2,-0.5 -1.4,3 -0.9,-0.1 z",
                    "CU" : "m 188.73,178.29 -3,-0.4 -1.1,-1.8 -2.4,-1 -2.6,-0.2 -0.1,-1 -1.7,-0.1 -3.3,1.9 0.2,-1.2 2.2,-1.3 2.3,-0.4 3.8,0.3 1.6,1.4 0.9,-0.3 0.8,0 3.4,2.7 3.5,1.1 0.9,1.1 -6.6,0.5 z",
                    "EC" : "m 182.53,224.19 1,-1.4 -2.1,-0.6 0,-2.4 1.6,-2.1 -0.1,-1.3 2.5,-1.4 2.6,1.5 2.2,0.4 1.9,1.1 -0.5,2.7 -2.1,2 -3.2,1.5 -1.4,3.1 -2.8,-1.8 z",
                    "AU" : "m 617.53,306.39 -1.6,0 -1.5,-2.2 -1.1,-3.5 0,-1.4 -1.2,-0.9 4.1,2.5 3.8,-1.3 0.1,2.7 -0.5,3.3 -0.9,-0.8 z m -1.4,-54.7 0.6,2.8 5.1,2.8 0.9,3.5 2.3,0.9 0.2,1.8 1.9,1.1 1.4,2.1 1.4,-0.3 -0.6,1.6 1,3.3 0.1,3 -2.1,7.5 -1.4,0.9 -3.1,6.5 -0.4,3.5 -3.2,0.7 -3.5,2 -2.9,-0.5 0.1,-1.3 -2.8,2.1 -2.2,-1.1 -3.3,-0.7 -1.8,-1.9 -0.3,-2.6 -1.1,-1 -4.4,1.1 1.2,-1 -0.6,-1.5 2,-3.8 -1.2,0 -3.4,3.9 -0.9,-2.3 -1.7,-1.5 -0.3,-1.5 -3.7,-0.9 -2.1,-1.2 -4.1,0.5 -3.2,1.2 -2.1,-0.1 -4,2 -1.1,1.6 -6.8,0 -3,2.1 -2.4,0.4 -2.3,-0.5 -1.9,-1.6 1.3,-1.7 0.1,-3.5 -1.4,-3 0,-1.6 -1.5,-2.8 -0.3,-1.6 -2.1,-3.2 1.1,-0.5 1.1,1.9 0.1,-1.4 -1.3,-2.5 0.4,-3.9 4.3,-3.1 3.5,-0.8 6.2,-2 2.6,-2.9 -0.3,-1.8 1.5,-1.3 1.1,1.8 0.1,-2.6 1.6,0.3 0,-2.1 1.3,-1.2 2.9,-0.8 0.6,-1 2.4,1.9 2.8,0.8 -0.6,-1.3 3.1,-4.8 -1.8,-0.3 0.7,-1.1 2,0.1 0,2 2.4,-0.3 -0.2,-2.2 1.1,1.3 3.7,1.1 2.7,-0.9 0.8,1 -1.9,2.8 -0.9,2.2 2.1,1.6 4.7,2.3 1.9,1.6 1.6,-0.5 1.6,-4.6 -0.1,-5 0.7,-0.9 0.3,-3.6 0.9,1.1 1.8,5.8 0.8,1.5 1,-0.7 1.7,1.5 0.1,2.7 z",
                    "VE" : "m 199.73,195.09 -1.1,0.5 0.6,1.7 -1,1.7 1.2,1.5 0.9,-1.3 -1.1,-2.1 2.8,-1.2 2.4,0 1.8,1.9 3.2,-0.4 2.9,1.1 1.6,-1 2.3,-0.3 0.9,1.4 2.8,1.3 -0.2,0.9 1.7,0.7 -1,1.4 0.4,1.3 -1.7,1.1 -0.3,1.2 1.2,1.4 -1.6,1.8 -5.9,0.5 1.2,2.9 1.3,0.1 -1.5,1.6 -2.7,1.7 -2.6,-1.1 -0.6,-2.2 -1.2,-0.9 1,-1 -1.1,-2.2 0.8,-3.1 -3.8,0.1 -1.3,-1.6 -3.8,-0.2 -0.5,-2.5 -1.2,-1.8 0.2,-2.1 1.8,-2.4 z",
                    "SB" : "m 644.23,236.59 -1.9,-0.1 0,-2.4 -2.9,-1.9 -1.4,0.8 1.2,1.3 -2.6,-1.1 0.1,-1.7 3.3,0.7 2.3,1.4 0.5,1.4 1.7,-0.8 z",
                    "MG" : "m 431.73,249.49 -4.3,14.5 -1,2.3 -3.7,1.2 -2.1,-1.2 -0.5,-3.1 -1.1,-2.4 0.6,-1.9 1.8,-2.6 -1,-5 1,-2.5 2.2,-0.4 3.6,-1.7 1.4,-3.3 0.9,0.1 1.1,-2.9 1.3,2.1 1,4.7 -1.6,0.2 z",
                    "IS" : "m 290.23,72.39 4.3,-0.5 -0.3,-1.1 -2.3,-0.6 -2.6,0.2 1.5,-0.8 0.6,-1.4 1.9,0.4 -0.2,-1.2 2.4,1.5 -0.4,1.4 1.1,0.6 1.6,-2.4 3.4,-0.2 1.4,0.7 2.1,-0.7 -0.1,-1 3.6,1.4 -0.3,1 2,0.7 0.5,1.4 -2,2.1 -6.1,2.1 -1.9,1 -2.8,-0.5 -1.9,-1 -2.9,0.1 1.9,-1.7 -1.4,-1.3 z",
                    "EG" : "m 403.13,159.99 -1.2,3.6 -2,-1.7 -1.4,-2.9 0.2,2.1 1.9,2.4 0.8,2.4 2.9,5.4 0.3,2.1 2.3,1.8 -10.8,0 -12,0 0,-14.6 -0.6,-2.1 0.9,-3.1 4.2,0.7 3.3,1.1 2.5,-1.4 2.3,0.1 1,0.8 4.1,-0.5 1.2,3.3 z",
                    "KG" : "m 477.33,138.49 -2.7,0.6 -1.5,-0.9 -4.1,0.2 1.2,-1.6 1.9,0 1.4,0.2 2.8,-1.5 -2.7,-1.5 -0.6,0.7 -2.4,-1 1.5,-1.5 1.7,-1.4 3.2,1 1.3,-2 1.9,1 6.2,-0.1 3.4,1.6 -3.5,1.9 -0.5,0.7 -2.5,0.3 -2.1,1.4 -0.3,-0.7 -3,1.3 z m -5.8,-0.9 -0.3,0 0.2,0.2 0.2,-0.1 z m 2.2,-0.1 0,-0.2 -0.2,0.1 0,0.1 z m -1.2,-0.3 -0.4,-0.1 0.2,0.5 0.3,-0.1 z",
                    "NP" : "m 505.03,163.39 0.1,2.3 -1.6,0.7 -2.9,-0.4 -2.4,-1.6 -3.5,-0.3 -5.2,-2.8 0.6,-1.9 1.3,-0.9 2,-0.3 2.8,2.4 1.1,-0.1 1.9,1.9 1.7,0.9 1.1,-0.4 z"
                }
            }
        }
    });

    return Mapael;

}));
